/*
 * Copyright (c) 2022-2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include <algorithm>
#include <cstddef>

namespace arm_conv {
namespace depthwise {

void sme2_fp32_planar_5x5_s1_4rows_mla_za_impl(
  const float *inptr,
  size_t ld_in_row,
  size_t ld_in_col,
  size_t ld_in_vl,
  unsigned int pad_top,
  unsigned int valid_input_rows,
  unsigned int pad_left,
  unsigned int valid_input_cols,
  const float *weights,
  const float *bias,
  float **outptrs,
  const size_t *outlds,
  const size_t *outvllds,
  unsigned int output_cols,
  unsigned int start_channel,
  unsigned int valid_channels,
  float act_min,
  float act_max
)
{
  struct Args
  {
    const float *inptr;
    size_t ld_in_vl;
    long unsigned int pad_top, pad_bottom, pad_left;
    const float *weights;
    const float *bias;
    long unsigned int input_cols, output_cols;
    float **outptrs;
    const size_t *ld_out_cols;
    const size_t *ld_out_vls;
    long unsigned int current_channel, n_channels;
    float clamp_min, clamp_max;
  };

  Args args = { inptr, ld_in_vl, pad_top, 8u - std::min(8u, pad_top + valid_input_rows), pad_left, weights, bias, valid_input_cols, output_cols, outptrs, outlds, outvllds, start_channel, valid_channels, act_min, act_max };

  __asm__ __volatile__(
    "ldr x6, [%x[args], %[offsetof_Args_pad_bottom]]\n"
    "mov x20, #0x8\n"
    ".inst 0xd503477f  // SMSTART ZA\n"
    "sub x20, x20, x6\n"
    "ldr x7, [%x[args], %[offsetof_Args_pad_top]]\n"
    "ptrue p2.b\n"
    ".inst 0x25207812  // ptrue pn10.b\n"
    "ld1rw { z16.s }, p2/Z, [%x[args], %[offsetof_Args_clamp_min]]\n"
    "ldr x17, [%x[args], %[offsetof_Args_n_channels]]\n"
    "whilelt p1.s, XZR, x17\n"
    "whilelt p9.s, XZR, x20\n"
    "ld1rw { z17.s }, p2/Z, [%x[args], %[offsetof_Args_clamp_max]]\n"
    "whilelt p8.s, XZR, x7\n"
    "eor p8.b, p2/Z, p8.b, p9.b\n"
    "ldr x16, [%x[args], %[offsetof_Args_current_channel]]\n"
    "1:"  // Channel loop
    "ldr x20, [%x[args], %[offsetof_Args_bias]]\n"
    "fmov z28.s, #0x0\n"
    "cbz x20, 2f\n"
    "ld1w { z28.s }, p1/Z, [x20, x16, LSL #2]\n"
    "2:"  // Load bias: Done
    "ldr x15, [%x[args], %[offsetof_Args_input_cols]]\n"
    "sub x20, x15, #0x1\n"
    "orr x23, x20, %x[ld_in_col], LSL #18\n"
    "mov z29.d, z28.d\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "orr x23, x17, x23, LSL #20\n"
    "mov x22, #0x8\n"
    "ldr x13, [%x[args], %[offsetof_Args_inptr]]\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "add x21, x7, x6\n"
    "lsl x20, %x[ld_in_row], #0x2\n"
    "ld1w { z2.s }, p2/Z, [x14, #4, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "mov z30.d, z28.d\n"
    "mov z31.d, z28.d\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    "mov x8, #0x0\n"
    "ldr x11, [%x[args], %[offsetof_Args_output_cols]]\n"
    "lsl x23, x23, #0x2\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "sub x22, x22, x21\n"
    "madd x20, x20, x7, x13\n"
    "ld1w { z3.s }, p2/Z, [x14, #4, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "3:"  // Issue prefetches
    "subs x22, x22, #0x1\n"
    ".inst 0xf8b74a9c  // rprfm pldstrm, x23, [x20]\n"
    "add x20, x20, %x[ld_in_col], LSL #2\n"
    "bgt 3b\n"
    "ldr x22, [%x[args], %[offsetof_Args_outptrs]]\n"
    "lsl x20, %x[ld_in_row], #0x2\n"
    "msub x13, x7, x20, x13\n"
    ".inst 0xc0040f80  // mova za.d[x8, #0], { z28.d-z31.d }\n"
    "ldr x20, [%x[args], %[offsetof_Args_ld_out_cols]]\n"
    ".inst 0xc0040f81  // mova za.d[x8, #1], { z28.d-z31.d }\n"
    "mov x10, #0x4\n"
    "ldp x9, x28, [x22], #0x10\n"
    ".inst 0xc0040f82  // mova za.d[x8, #2], { z28.d-z31.d }\n"
    "ldp x27, x26, [x20], #0x10\n"
    ".inst 0xc0040f83  // mova za.d[x8, #3], { z28.d-z31.d }\n"
    "ldr x21, [%x[args], %[offsetof_Args_pad_left]]\n"
    ".inst 0xc0040f84  // mova za.d[x8, #4], { z28.d-z31.d }\n"
    "ldp x25, x24, [x22], #0x10\n"
    "ldp x23, x22, [x20], #0x10\n"
    "cbz x21, 5f\n"
    "cmp x21, x10\n"
    "csel x20, x21, x10, LT\n"
    "sub x21, x21, x20\n"
    "sub x10, x10, x20\n"
    "cbz x21, 5f\n"
    ".inst 0xc0060c04  // mova { z4.d-z7.d }, za.d[x8, #0]\n"
    "sub x11, x11, x21\n"
    ".inst 0xc1b1ca04  // fclamp { z4.s-z7.s }, z16.s, z17.s\n"
    "4:"  // Left padding
    "subs x21, x21, #0x1\n"
    "st1w { z4.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    "st1w { z5.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    "st1w { z6.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    "st1w { z7.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    "bgt 4b\n"
    "5:"  // Left padding: End
    "adds XZR, x7, x6\n"
    "bne 12f\n"
    "cbz x10, 10f\n"
    "cmp x10, #0x1\n"
    "sub x15, x15, x10\n"
    "beq 9f\n"
    "cmp x10, #0x2\n"
    "beq 8f\n"
    "cmp x10, #0x3\n"
    "beq 7f\n"
    "6:"  // Unpadded: 4 priming loads
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z18.s }, p1/Z, [x13]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z19.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13e1a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z14.s\n"
    "ld1w { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13a1a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z10.s\n"
    "ld1w { z23.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1351a80  // fmla za.s[x8, 0], { z20.s-z23.s }, z5.s\n"
    "ld1w { z24.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa04049cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13c1aa0  // fmla za.s[x8, 0], { z21.s-z24.s }, z12.s\n"
    "ld1w { z25.s }, p1/Z, [x20]\n"
    ".inst 0xa14049c1  // ld1w { z1.s, z9.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1311ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z1.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "7:"  // Unpadded: 3 priming loads
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z20.s }, p1/Z, [x13]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z23.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13f1a80  // fmla za.s[x8, 0], { z20.s-z23.s }, z15.s\n"
    ".inst 0xc13e1a81  // fmla za.s[x8, 1], { z20.s-z23.s }, z14.s\n"
    "ld1w { z24.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13b1aa0  // fmla za.s[x8, 0], { z21.s-z24.s }, z11.s\n"
    "ld1w { z25.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13a1aa1  // fmla za.s[x8, 1], { z21.s-z24.s }, z10.s\n"
    ".inst 0xa04049c6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1371ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z7.s\n"
    "ld1w { z26.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1361ac1  // fmla za.s[x8, 1], { z22.s-z25.s }, z6.s\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13d1ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z13.s\n"
    "ld1w { z27.s }, p1/Z, [x20]\n"
    ".inst 0xc1351ae1  // fmla za.s[x8, 1], { z23.s-z26.s }, z5.s\n"
    ".inst 0xa04049c6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1371b00  // fmla za.s[x8, 0], { z24.s-z27.s }, z7.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1361b01  // fmla za.s[x8, 1], { z24.s-z27.s }, z6.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "8:"  // Unpadded: 2 priming loads
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z1.s }, p1/Z, [x13]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z2.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z3.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z4.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13c1820  // fmla za.s[x8, 0], { z1.s-z4.s }, z12.s\n"
    ".inst 0xc13f1821  // fmla za.s[x8, 1], { z1.s-z4.s }, z15.s\n"
    "ld1w { z5.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13e1822  // fmla za.s[x8, 2], { z1.s-z4.s }, z14.s\n"
    "ld1w { z6.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1381840  // fmla za.s[x8, 0], { z2.s-z5.s }, z8.s\n"
    ".inst 0xa04049cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14]\n"
    ".inst 0xc13b1841  // fmla za.s[x8, 1], { z2.s-z5.s }, z11.s\n"
    ".inst 0xa04149ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13a1842  // fmla za.s[x8, 2], { z2.s-z5.s }, z10.s\n"
    "ld1w { z7.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13e1860  // fmla za.s[x8, 0], { z3.s-z6.s }, z14.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    ".inst 0xc13d1861  // fmla za.s[x8, 1], { z3.s-z6.s }, z13.s\n"
    ".inst 0xa14149c0  // ld1w { z0.s, z8.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13c1862  // fmla za.s[x8, 2], { z3.s-z6.s }, z12.s\n"
    "ld1w { z8.s }, p1/Z, [x20]\n"
    ".inst 0xc1301880  // fmla za.s[x8, 0], { z4.s-z7.s }, z0.s\n"
    ".inst 0xa04049c0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x14]\n"
    ".inst 0xc13f1881  // fmla za.s[x8, 1], { z4.s-z7.s }, z15.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13e1882  // fmla za.s[x8, 2], { z4.s-z7.s }, z14.s\n"
    ".inst 0xc13c18a0  // fmla za.s[x8, 0], { z5.s-z8.s }, z12.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13118a1  // fmla za.s[x8, 1], { z5.s-z8.s }, z1.s\n"
    ".inst 0xc13018a2  // fmla za.s[x8, 2], { z5.s-z8.s }, z0.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "9:"  // Unpadded: 1 priming loads
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z20.s }, p1/Z, [x13]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z23.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13d1a80  // fmla za.s[x8, 0], { z20.s-z23.s }, z13.s\n"
    ".inst 0xc13c1a81  // fmla za.s[x8, 1], { z20.s-z23.s }, z12.s\n"
    "ld1w { z24.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13f1a82  // fmla za.s[x8, 2], { z20.s-z23.s }, z15.s\n"
    "ld1w { z25.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13e1a83  // fmla za.s[x8, 3], { z20.s-z23.s }, z14.s\n"
    ".inst 0xa04049c4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x14]\n"
    ".inst 0xc1391aa0  // fmla za.s[x8, 0], { z21.s-z24.s }, z9.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1381aa1  // fmla za.s[x8, 1], { z21.s-z24.s }, z8.s\n"
    "ld1w { z26.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13b1aa2  // fmla za.s[x8, 2], { z21.s-z24.s }, z11.s\n"
    ".inst 0xa14149c6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13a1aa3  // fmla za.s[x8, 3], { z21.s-z24.s }, z10.s\n"
    ".inst 0xa14049c1  // ld1w { z1.s, z9.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13d1ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z13.s\n"
    "ld1w { z27.s }, p1/Z, [x20]\n"
    ".inst 0xc13c1ac1  // fmla za.s[x8, 1], { z22.s-z25.s }, z12.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1351ac2  // fmla za.s[x8, 2], { z22.s-z25.s }, z5.s\n"
    ".inst 0xc1341ac3  // fmla za.s[x8, 3], { z22.s-z25.s }, z4.s\n"
    ".inst 0xa04049c4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13e1ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z14.s\n"
    "ld1w { z2.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1361ae1  // fmla za.s[x8, 1], { z23.s-z26.s }, z6.s\n"
    ".inst 0xc1391ae2  // fmla za.s[x8, 2], { z23.s-z26.s }, z9.s\n"
    ".inst 0xc1311ae3  // fmla za.s[x8, 3], { z23.s-z26.s }, z1.s\n"
    ".inst 0xc13d1b00  // fmla za.s[x8, 0], { z24.s-z27.s }, z13.s\n"
    ".inst 0xc13c1b01  // fmla za.s[x8, 1], { z24.s-z27.s }, z12.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1351b02  // fmla za.s[x8, 2], { z24.s-z27.s }, z5.s\n"
    ".inst 0xc1341b03  // fmla za.s[x8, 3], { z24.s-z27.s }, z4.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "ld1w { z3.s }, p2/Z, [x14, #4, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "10:"  // Unpadded: 0 priming loads
    "cbz x15, 20f\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z18.s }, p1/Z, [x13]\n"
    "sub x15, x15, #0x1\n"
    "ld1w { z19.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "sub x11, x11, #0x1\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "cmp x15, x11\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "csel x21, x15, x11, LT\n"
    "ld1w { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z23.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "sub x11, x11, x21\n"
    "ld1w { z24.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z25.s }, p1/Z, [x20]\n"
    "cbz x21, 19f\n"
    "11:"  // Unpadded: Main loop
    ".inst 0xc1321a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z2.s\n"
    "ld1w { z6.s }, p2/Z, [x14, #4, MUL VL]\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "subs x21, x21, #0x1\n"
    ".inst 0xc1331a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z3.s\n"
    ".inst 0xc13d1a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z13.s\n"
    ".inst 0xc13c1a42  // fmla za.s[x8, 2], { z18.s-z21.s }, z12.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13f1a43  // fmla za.s[x8, 3], { z18.s-z21.s }, z15.s\n"
    ".inst 0xc13e1a44  // fmla za.s[x8, 4], { z18.s-z21.s }, z14.s\n"
    ".inst 0xa04049c4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1361a80  // fmla za.s[x8, 0], { z20.s-z23.s }, z6.s\n"
    "ld1w { z6.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1391a61  // fmla za.s[x8, 1], { z19.s-z22.s }, z9.s\n"
    "ld1w { z18.s }, p1/Z, [x13]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    ".inst 0xc1381a62  // fmla za.s[x8, 2], { z19.s-z22.s }, z8.s\n"
    ".inst 0xa04149ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13b1a63  // fmla za.s[x8, 3], { z19.s-z22.s }, z11.s\n"
    ".inst 0xc13a1a64  // fmla za.s[x8, 4], { z19.s-z22.s }, z10.s\n"
    ".inst 0xa14049c0  // ld1w { z0.s, z8.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1361aa0  // fmla za.s[x8, 0], { z21.s-z24.s }, z6.s\n"
    "ld1w { z2.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc13d1a81  // fmla za.s[x8, 1], { z20.s-z23.s }, z13.s\n"
    "ld1w { z19.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13c1a82  // fmla za.s[x8, 2], { z20.s-z23.s }, z12.s\n"
    ".inst 0xa04149c6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1351a83  // fmla za.s[x8, 3], { z20.s-z23.s }, z5.s\n"
    ".inst 0xc1341a84  // fmla za.s[x8, 4], { z20.s-z23.s }, z4.s\n"
    ".inst 0xa04049c4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1321ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z2.s\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13f1aa1  // fmla za.s[x8, 1], { z21.s-z24.s }, z15.s\n"
    "ld1w { z2.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc13e1aa2  // fmla za.s[x8, 2], { z21.s-z24.s }, z14.s\n"
    ".inst 0xc1381aa3  // fmla za.s[x8, 3], { z21.s-z24.s }, z8.s\n"
    ".inst 0xc1301aa4  // fmla za.s[x8, 4], { z21.s-z24.s }, z0.s\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc0060c0c  // mova { z12.d-z15.d }, za.d[x8, #0]\n"
    ".inst 0xc1b1ca0c  // fclamp { z12.s-z15.s }, z16.s, z17.s\n"
    "st1w { z12.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    ".inst 0xc1371ac1  // fmla za.s[x8, 1], { z22.s-z25.s }, z7.s\n"
    "st1w { z13.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    ".inst 0xc1361ac2  // fmla za.s[x8, 2], { z22.s-z25.s }, z6.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "st1w { z14.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    ".inst 0xc1351ac3  // fmla za.s[x8, 3], { z22.s-z25.s }, z5.s\n"
    "st1w { z15.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    ".inst 0xc1341ac4  // fmla za.s[x8, 4], { z22.s-z25.s }, z4.s\n"
    "ld1w { z22.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "add x8, x8, #0x1\n"
    "ld1w { z23.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc0040f84  // mova za.d[x8, #4], { z28.d-z31.d }\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z24.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "ld1w { z3.s }, p2/Z, [x14, #4, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z25.s }, p1/Z, [x20]\n"
    "bgt 11b\n"
    "b 19f\n"
    "12:"  // Padded
    "cbz x10, 17f\n"
    "cmp x10, #0x1\n"
    "sub x15, x15, x10\n"
    "beq 16f\n"
    "cmp x10, #0x2\n"
    "beq 15f\n"
    "cmp x10, #0x3\n"
    "beq 14f\n"
    "13:"  // Padded: 4 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z19.s }, p0/Z, [x13]\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1w { z22.s }, p0/Z, [x20]\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13e1a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z14.s\n"
    ".inst 0xa14049c1  // ld1w { z1.s, z9.s }, pn10.b/Z, [x14]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z23.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13a1a80  // fmla za.s[x8, 0], { z20.s-z23.s }, z10.s\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z24.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1311aa0  // fmla za.s[x8, 0], { z21.s-z24.s }, z1.s\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z25.s }, p0/Z, [x20]\n"
    ".inst 0xa04049c6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x14]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xc1361ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z6.s\n"
    "addvl x14, x14, #5\n"
    "ld1w { z26.s }, p0/Z, [x20]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    ".inst 0xa14049c0  // ld1w { z0.s, z8.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1301ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z0.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "14:"  // Padded: 3 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z0.s }, p0/Z, [x13]\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z1.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z2.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1w { z3.s }, p0/Z, [x20]\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13f1800  // fmla za.s[x8, 0], { z0.s-z3.s }, z15.s\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc13e1801  // fmla za.s[x8, 1], { z0.s-z3.s }, z14.s\n"
    "ld1w { z4.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    ".inst 0xc13b1820  // fmla za.s[x8, 0], { z1.s-z4.s }, z11.s\n"
    "ld1w { z5.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13a1821  // fmla za.s[x8, 1], { z1.s-z4.s }, z10.s\n"
    ".inst 0xa04049c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1391840  // fmla za.s[x8, 0], { z2.s-z5.s }, z9.s\n"
    "ld1w { z6.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xc1381841  // fmla za.s[x8, 1], { z2.s-z5.s }, z8.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    ".inst 0xc13f1860  // fmla za.s[x8, 0], { z3.s-z6.s }, z15.s\n"
    "ld1w { z7.s }, p0/Z, [x20]\n"
    ".inst 0xc13e1861  // fmla za.s[x8, 1], { z3.s-z6.s }, z14.s\n"
    ".inst 0xa14049c3  // ld1w { z3.s, z11.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13b1880  // fmla za.s[x8, 0], { z4.s-z7.s }, z11.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1331881  // fmla za.s[x8, 1], { z4.s-z7.s }, z3.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "15:"  // Padded: 2 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z19.s }, p0/Z, [x13]\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1w { z22.s }, p0/Z, [x20]\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13c1a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z12.s\n"
    ".inst 0xa04149c6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc13f1a61  // fmla za.s[x8, 1], { z19.s-z22.s }, z15.s\n"
    "ld1w { z23.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13e1a62  // fmla za.s[x8, 2], { z19.s-z22.s }, z14.s\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z24.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1381a80  // fmla za.s[x8, 0], { z20.s-z23.s }, z8.s\n"
    ".inst 0xa14049c0  // ld1w { z0.s, z8.s }, pn10.b/Z, [x14]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13b1a81  // fmla za.s[x8, 1], { z20.s-z23.s }, z11.s\n"
    "ld1w { z25.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xc13a1a82  // fmla za.s[x8, 2], { z20.s-z23.s }, z10.s\n"
    ".inst 0xa14049c2  // ld1w { z2.s, z10.s }, pn10.b/Z, [x14]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    ".inst 0xc1361aa0  // fmla za.s[x8, 0], { z21.s-z24.s }, z6.s\n"
    ".inst 0xa14149c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1381aa1  // fmla za.s[x8, 1], { z21.s-z24.s }, z8.s\n"
    "ld1w { z26.s }, p0/Z, [x20]\n"
    ".inst 0xc1301aa2  // fmla za.s[x8, 2], { z21.s-z24.s }, z0.s\n"
    ".inst 0xa04049c6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x14]\n"
    ".inst 0xc1341ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z4.s\n"
    ".inst 0xa14149c3  // ld1w { z3.s, z11.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13a1ac1  // fmla za.s[x8, 1], { z22.s-z25.s }, z10.s\n"
    ".inst 0xc1321ac2  // fmla za.s[x8, 2], { z22.s-z25.s }, z2.s\n"
    ".inst 0xc1331ae0  // fmla za.s[x8, 0], { z23.s-z26.s }, z3.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1371ae1  // fmla za.s[x8, 1], { z23.s-z26.s }, z7.s\n"
    ".inst 0xc1361ae2  // fmla za.s[x8, 2], { z23.s-z26.s }, z6.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "16:"  // Padded: 1 priming loads
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z18.s }, p0/Z, [x13]\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z19.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13d1a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z13.s\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc13c1a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z12.s\n"
    "ld1w { z22.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13f1a42  // fmla za.s[x8, 2], { z18.s-z21.s }, z15.s\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z23.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13e1a43  // fmla za.s[x8, 3], { z18.s-z21.s }, z14.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    ".inst 0xc1391a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z9.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1381a61  // fmla za.s[x8, 1], { z19.s-z22.s }, z8.s\n"
    "ld1w { z24.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xc13b1a62  // fmla za.s[x8, 2], { z19.s-z22.s }, z11.s\n"
    ".inst 0xa14149c0  // ld1w { z0.s, z8.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13a1a63  // fmla za.s[x8, 3], { z19.s-z22.s }, z10.s\n"
    ".inst 0xa14049c1  // ld1w { z1.s, z9.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13d1a80  // fmla za.s[x8, 0], { z20.s-z23.s }, z13.s\n"
    "ld1w { z25.s }, p0/Z, [x20]\n"
    ".inst 0xc13c1a81  // fmla za.s[x8, 1], { z20.s-z23.s }, z12.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13f1a82  // fmla za.s[x8, 2], { z20.s-z23.s }, z15.s\n"
    ".inst 0xc13e1a83  // fmla za.s[x8, 3], { z20.s-z23.s }, z14.s\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1381aa0  // fmla za.s[x8, 0], { z21.s-z24.s }, z8.s\n"
    "ld1w { z2.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1301aa1  // fmla za.s[x8, 1], { z21.s-z24.s }, z0.s\n"
    ".inst 0xc1391aa2  // fmla za.s[x8, 2], { z21.s-z24.s }, z9.s\n"
    ".inst 0xc1311aa3  // fmla za.s[x8, 3], { z21.s-z24.s }, z1.s\n"
    ".inst 0xc13d1ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z13.s\n"
    ".inst 0xc13c1ac1  // fmla za.s[x8, 1], { z22.s-z25.s }, z12.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13b1ac2  // fmla za.s[x8, 2], { z22.s-z25.s }, z11.s\n"
    ".inst 0xc13a1ac3  // fmla za.s[x8, 3], { z22.s-z25.s }, z10.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "ld1w { z3.s }, p2/Z, [x14, #4, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "17:"  // Padded: 0 priming loads
    "cbz x15, 20f\n"
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z18.s }, p0/Z, [x13]\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z19.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    "mov x12, #0x4\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z22.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "sub x15, x15, #0x1\n"
    "ld1w { z23.s }, p0/Z, [x20]\n"
    "sub x11, x11, #0x1\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "cmp x15, x11\n"
    "ld1w { z24.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "ld1w { z25.s }, p0/Z, [x20]\n"
    "csel x21, x15, x11, LT\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "sub x11, x11, x21\n"
    "cbz x21, 19f\n"
    "18:"  // Padded: Main loop
    ".inst 0xc1321a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z2.s\n"
    "ld1w { z0.s }, p2/Z, [x14, #4, MUL VL]\n"
    "mov x12, #0x0\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc1331a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z3.s\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "subs x21, x21, #0x1\n"
    ".inst 0xc13d1a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z13.s\n"
    ".inst 0xc13c1a42  // fmla za.s[x8, 2], { z18.s-z21.s }, z12.s\n"
    ".inst 0xa04149c2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13f1a43  // fmla za.s[x8, 3], { z18.s-z21.s }, z15.s\n"
    ".inst 0xc13e1a44  // fmla za.s[x8, 4], { z18.s-z21.s }, z14.s\n"
    ".inst 0xa04049c4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1301a80  // fmla za.s[x8, 0], { z20.s-z23.s }, z0.s\n"
    "ld1w { z12.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1391a61  // fmla za.s[x8, 1], { z19.s-z22.s }, z9.s\n"
    "ld1w { z18.s }, p0/Z, [x13]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    ".inst 0xc1381a62  // fmla za.s[x8, 2], { z19.s-z22.s }, z8.s\n"
    ".inst 0xa14149c0  // ld1w { z0.s, z8.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13b1a63  // fmla za.s[x8, 3], { z19.s-z22.s }, z11.s\n"
    ".inst 0xc13a1a64  // fmla za.s[x8, 4], { z19.s-z22.s }, z10.s\n"
    ".inst 0xa04049c6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13c1aa0  // fmla za.s[x8, 0], { z21.s-z24.s }, z12.s\n"
    "ld1w { z12.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1331a81  // fmla za.s[x8, 1], { z20.s-z23.s }, z3.s\n"
    "ld1w { z19.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xc1321a82  // fmla za.s[x8, 2], { z20.s-z23.s }, z2.s\n"
    ".inst 0xa14149c3  // ld1w { z3.s, z11.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1351a83  // fmla za.s[x8, 3], { z20.s-z23.s }, z5.s\n"
    ".inst 0xc1341a84  // fmla za.s[x8, 4], { z20.s-z23.s }, z4.s\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "mov x12, #0x4\n"
    ".inst 0xc13c1ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z12.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1381aa1  // fmla za.s[x8, 1], { z21.s-z24.s }, z8.s\n"
    "ld1w { z2.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1301aa2  // fmla za.s[x8, 2], { z21.s-z24.s }, z0.s\n"
    ".inst 0xc1371aa3  // fmla za.s[x8, 3], { z21.s-z24.s }, z7.s\n"
    ".inst 0xc1361aa4  // fmla za.s[x8, 4], { z21.s-z24.s }, z6.s\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    ".inst 0xc0060c04  // mova { z4.d-z7.d }, za.d[x8, #0]\n"
    ".inst 0xc1b1ca04  // fclamp { z4.s-z7.s }, z16.s, z17.s\n"
    "st1w { z4.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    ".inst 0xc13b1ac1  // fmla za.s[x8, 1], { z22.s-z25.s }, z11.s\n"
    "st1w { z5.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    ".inst 0xc1331ac2  // fmla za.s[x8, 2], { z22.s-z25.s }, z3.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "st1w { z6.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    ".inst 0xc13f1ac3  // fmla za.s[x8, 3], { z22.s-z25.s }, z15.s\n"
    "st1w { z7.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    ".inst 0xc13e1ac4  // fmla za.s[x8, 4], { z22.s-z25.s }, z14.s\n"
    "ld1w { z22.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "ld1w { z23.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "add x8, x8, #0x1\n"
    ".inst 0xc0040f84  // mova za.d[x8, #4], { z28.d-z31.d }\n"
    "ld1w { z24.s }, p0/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "ld1w { z3.s }, p2/Z, [x14, #4, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z25.s }, p0/Z, [x20]\n"
    "bgt 18b\n"
    "19:"  // Main loop tail
    ".inst 0xc1321a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z2.s\n"
    "ld1w { z6.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1331a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z3.s\n"
    ".inst 0xc13d1a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z13.s\n"
    ".inst 0xc13c1a42  // fmla za.s[x8, 2], { z18.s-z21.s }, z12.s\n"
    ".inst 0xa04149c4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13f1a43  // fmla za.s[x8, 3], { z18.s-z21.s }, z15.s\n"
    ".inst 0xc13e1a44  // fmla za.s[x8, 4], { z18.s-z21.s }, z14.s\n"
    ".inst 0xa04049c2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1361a80  // fmla za.s[x8, 0], { z20.s-z23.s }, z6.s\n"
    "ld1w { z7.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1391a61  // fmla za.s[x8, 1], { z19.s-z22.s }, z9.s\n"
    ".inst 0xc1381a62  // fmla za.s[x8, 2], { z19.s-z22.s }, z8.s\n"
    ".inst 0xa14149c0  // ld1w { z0.s, z8.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13b1a63  // fmla za.s[x8, 3], { z19.s-z22.s }, z11.s\n"
    ".inst 0xc13a1a64  // fmla za.s[x8, 4], { z19.s-z22.s }, z10.s\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1371aa0  // fmla za.s[x8, 0], { z21.s-z24.s }, z7.s\n"
    "ld1w { z1.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1351a81  // fmla za.s[x8, 1], { z20.s-z23.s }, z5.s\n"
    ".inst 0xc1341a82  // fmla za.s[x8, 2], { z20.s-z23.s }, z4.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1331a83  // fmla za.s[x8, 3], { z20.s-z23.s }, z3.s\n"
    ".inst 0xc1321a84  // fmla za.s[x8, 4], { z20.s-z23.s }, z2.s\n"
    ".inst 0xa04049c2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x14]\n"
    ".inst 0xc1311ac0  // fmla za.s[x8, 0], { z22.s-z25.s }, z1.s\n"
    ".inst 0xc1381aa1  // fmla za.s[x8, 1], { z21.s-z24.s }, z8.s\n"
    ".inst 0xc1301aa2  // fmla za.s[x8, 2], { z21.s-z24.s }, z0.s\n"
    ".inst 0xc13b1aa3  // fmla za.s[x8, 3], { z21.s-z24.s }, z11.s\n"
    ".inst 0xc13a1aa4  // fmla za.s[x8, 4], { z21.s-z24.s }, z10.s\n"
    ".inst 0xc0060c04  // mova { z4.d-z7.d }, za.d[x8, #0]\n"
    ".inst 0xc1b1ca04  // fclamp { z4.s-z7.s }, z16.s, z17.s\n"
    "st1w { z4.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    ".inst 0xc13d1ac1  // fmla za.s[x8, 1], { z22.s-z25.s }, z13.s\n"
    "st1w { z5.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    ".inst 0xc13c1ac2  // fmla za.s[x8, 2], { z22.s-z25.s }, z12.s\n"
    "st1w { z6.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    ".inst 0xc1331ac3  // fmla za.s[x8, 3], { z22.s-z25.s }, z3.s\n"
    "st1w { z7.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    ".inst 0xc1321ac4  // fmla za.s[x8, 4], { z22.s-z25.s }, z2.s\n"
    "add x8, x8, #0x1\n"
    ".inst 0xc0040f84  // mova za.d[x8, #4], { z28.d-z31.d }\n"
    "20:"  // Main loop skip tail
    "cbz x11, 22f\n"
    "21:"  // Right padding loop
    ".inst 0xc0060c00  // mova { z0.d-z3.d }, za.d[x8, #0]\n"
    "add x8, x8, #0x1\n"
    "subs x11, x11, #0x1\n"
    ".inst 0xc1b1ca00  // fclamp { z0.s-z3.s }, z16.s, z17.s\n"
    "st1w { z0.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    ".inst 0xc0040f84  // mova za.d[x8, #4], { z28.d-z31.d }\n"
    "st1w { z1.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    "st1w { z2.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    "st1w { z3.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    "bgt 21b\n"
    "22:"  // End
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    "incb x20, ALL, MUL #16\n"
    "incb x20, ALL, MUL #9\n"
    "str x20, [%x[args], %[offsetof_Args_weights]]\n"
    "ldr x21, [%x[args], %[offsetof_Args_ld_in_vl]]\n"
    "incw x16\n"
    "whilelt p1.s, x16, x17\n"
    "ldr x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "add x20, x20, x21, LSL #2\n"
    "str x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "ldr x25, [%x[args], %[offsetof_Args_outptrs]]\n"
    "ldr x24, [%x[args], %[offsetof_Args_ld_out_vls]]\n"
    "ldp x23, x22, [x25, #0x0]\n"
    "ldp x21, x20, [x24, #0x0]\n"
    "add x23, x23, x21, LSL #2\n"
    "add x22, x22, x20, LSL #2\n"
    "stp x23, x22, [x25, #0x0]\n"
    "ldp x23, x22, [x25, #0x10]\n"
    "ldp x21, x20, [x24, #0x10]\n"
    "add x23, x23, x21, LSL #2\n"
    "add x22, x22, x20, LSL #2\n"
    "stp x23, x22, [x25, #0x10]\n"
    "b.any 1b\n"
    ".inst 0xd503467f  // SMSTOP\n"
    :
    : [args] "r" (&args), [ld_in_col] "r" (ld_in_col), [ld_in_row] "r" (ld_in_row), [offsetof_Args_bias] "I" (offsetof(Args, bias)), [offsetof_Args_clamp_max] "I" (offsetof(Args, clamp_max)), [offsetof_Args_clamp_min] "I" (offsetof(Args, clamp_min)), [offsetof_Args_current_channel] "I" (offsetof(Args, current_channel)), [offsetof_Args_inptr] "I" (offsetof(Args, inptr)), [offsetof_Args_input_cols] "I" (offsetof(Args, input_cols)), [offsetof_Args_ld_in_vl] "I" (offsetof(Args, ld_in_vl)), [offsetof_Args_ld_out_cols] "I" (offsetof(Args, ld_out_cols)), [offsetof_Args_ld_out_vls] "I" (offsetof(Args, ld_out_vls)), [offsetof_Args_n_channels] "I" (offsetof(Args, n_channels)), [offsetof_Args_outptrs] "I" (offsetof(Args, outptrs)), [offsetof_Args_output_cols] "I" (offsetof(Args, output_cols)), [offsetof_Args_pad_bottom] "I" (offsetof(Args, pad_bottom)), [offsetof_Args_pad_left] "I" (offsetof(Args, pad_left)), [offsetof_Args_pad_top] "I" (offsetof(Args, pad_top)), [offsetof_Args_weights] "I" (offsetof(Args, weights))
    : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
  );
}

}  // namespace depthwise
}  // namespace arm_conv

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
