/*
 *  @(#)TransitionImpl.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.mbtf.v1.engine;


import net.sourceforge.groboutils.mbtf.v1.IState;
import net.sourceforge.groboutils.mbtf.v1.IAction;
import net.sourceforge.groboutils.mbtf.v1.IValidate;
import net.sourceforge.groboutils.mbtf.v1.ITransition;


/**
 * Nearly-Immutable implementation of ITransition.  As an immutable, transitions
 * cannot be created if there are cycles in the state machine.  Therefore,
 * there needs to be a way to post-creation correctly populate the Transition's
 * destination state.  Thus, the destination state may be set once.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:52:26 $
 * @since      June 12, 2002
 */
public class TransitionImpl implements ITransition
{
    private String name;
    private IState destination;
    private IAction action;
    private IValidate[] validate;
    
    
    public TransitionImpl( String name, IState dest, IAction a, IValidate[] v )
    {
        // allow destination to be null

        
        if (name == null || a == null)
        {
            throw new IllegalArgumentException("no null args");
        }
        
        if (v == null)
        {
            v = new IValidate[0];
        }
        else
        {
            int len = v.length;
            IValidate[] vv = new IValidate[ len ];
            for (int i = len; --i >= 0;)
            {
                if (v[i] == null)
                {
                    throw new IllegalArgumentException(
                        "no nulls allowed in IValidate array");
                }
                // else
                vv[i] = v[i];
            }
            v = vv;
        }
        
        this.name = name;
        setDestinationState( dest );
        this.action = a;
        this.validate = v;
    }
    
    
    /**
     * Retrieves the name for the transition.  This should be unique for
     * state-machine assembly purposes, but it does not have to be.
     *
     * @return a non-null name for this transition.
     */
    public String getName()
    {
        return this.name;
    }
    
    
    /**
     * Returns the next state if the corresponding action is executed.
     *
     * @return destination state, which can never be <tt>null</tt>.
     */
    public IState getDestinationState()
    {
        if (isDestinationStateSet())
        {
            return this.destination;
        }
        // else
        throw new IllegalStateException("Destination was never set.");
    }
    
    
    /**
     * Returns the action used to transition to the destination state.
     *
     * @return transition's action, which can never be <tt>null</tt>.
     */
    public IAction getAction()
    {
        return this.action;
    }
    
    
    /**
     * Returns all validation methods used to assert that the system can
     * perform this transition.
     *
     * @return a list of associated validation instances.
     */
    public IValidate[] getValidates()
    {
        int len = this.validate.length;
        IValidate v[] = new IValidate[ len ];
        System.arraycopy( this.validate, 0, v, 0, len );
        return v;
    }
    
    //-------------------------------------------------------------------------
    
    /**
     * Allows for post-creation setting of the destination.  The destination
     * may only be set to a non-<tt>null</tt> value once.  If there is an
     * attempt to pull the destination through <tt>getDestinationState()</tt>
     * and it has not been set yet, then an IllegalStateException will
     * be thrown.
     */
    public void setDestinationState( IState dest )
    {
        if (isDestinationStateSet())
        {
            throw new IllegalStateException(
                "Destination has already been set.");
        }
        else
        {
            this.destination = dest;
        }
    }
    
    
    /**
     * Allows for a builder system to detect if the destination state has
     * been set or not, without causing the IllegalStateException through
     * the <tt>getDestinationState()</tt>.
     */
    public boolean isDestinationStateSet()
    {
        return (this.destination != null);
    }
    
    
    public String toString()
    {
        return "[Transition "+getName()+"]";
    }
}

