/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * See the AUTHORS file for a list of people who have hacked on 
 * this code. 
 * See the ChangeLog file for a list of changes.
 *
 * Contents:
 *
 *   ppd_file_free()       - Free all memory used by the PPD file.
 *   ppd_file_new()             - Initialize the PpdFile struct.  
 *   ppd_file_new_from_filep() - Initialize the PpdFile struct, 
 *                               use a FILE* reference to the ppd
 *   ppd_file_new_from_fd() - Initialize the PpdFile struct, use 
 *                            a file descriptor to the ppd
 *   ppd_free_group()  - Free a single UI group.
 *   ppd_free_option() - Free a single option.
 *   ppd_free_choice() - Free a single choice <mfasheh@valinux.com>
 *
 */

#ifndef __PPD_H_
#define __PPD_H_

#include <stdio.h>
#include <unistd.h>
#include <glib.h>

#ifndef DEBUG
//#define DEBUG
#endif

#include "ppdenums.h"
#include "ppdmacros.h"

// C++ magic...
#ifdef __cplusplus
extern "C" {
#endif				// __cplusplus
// PPD version.../* Kept in sync with Adobe version number */
#define PPD_VERSION	4.3
/* This is a virtual class.  It should never be instantiated. */ typedef struct {
    guint8 id;
  } PpdObject;

  // Types and structures... version numbererivatives) is governed
  typedef struct {		// *** Options ***
    guint8 id;			// Object id

    gboolean conflicted;	// 0 if no conflicts exist, 1 otherwise 
    gboolean emitted;		// 1 already emitted 0 otherwise

    GString *keyword;		// Option keyword name ("PageSize", etc.) 
    GString *defchoice;		// Default option choice 
    GString *text;		// Human-readable text

    PpdUIType ui;		// Type of UI option 
    PpdSectionOrder section;	// Section for command 
    float order;		// Order number
    GSList *choices;		// should be a GSList of PpdChoice objects
  } PpdOption;			/* was ppd_option_t */

  typedef struct {		// *** Option choices ***
    guint8 id;			// Object id

    gboolean marked;		// 0 if not selected, 1 otherwise
    GString *choice;		// Computer-readable option name: 41 chars 
    GString *text;		// Human-readable option name: 81 chars
    char *code;			// Code to send for this option 
    PpdOption *option;		// Pointer to parent option structure 
  } PpdChoice;			/* was ppd_choice_t */


  /* hrm..  this whole struct should be a GSList... */
  typedef struct {		// *** Groups ***
    guint8 id;			// Object id

    GString *text;		// Human-readable group name
    GSList *options;		/* should be a GSList of PpdOption objects */
    GSList *subgroups;		// GSList of PpdGroups (subgroups)
  } PpdGroup;			/* was ppd_group_t */

  typedef struct {		// *** Constraints ***
    guint8 id;			// Object id

    GString *option1;		// First keyword 
    GString *choice1;		// First option/choice (blank for all) 
    GString *option2;		// Second keyword 
    GString *choice2;		// Second option/choice (blank for all) 
  } PpdConstraint;		/* was ppd_const_t */

  typedef struct {		// *** Page Sizes ***
    guint8 id;			// Object id

    gboolean marked;		// Page size selected?
    GString *name;		// Media size option 
    float width;		// Width of media in points 
    float length;		// Length of media in points 
    float left;			// Left printable margin in points 
    float bottom;		// Bottom printable margin in points 
    float right;		// Right printable margin in points 
    float top;			// Top printable margin in points 
  } PpdSize;			/* ppd_size_t; */

  typedef struct {		// *** Emulators ***
    guint8 id;			// Object id

    GString *name;		// Emulator name 
    char *start;		// Code to switch to this emulation 
    char *stop;			// Code to stop this emulation 
  } PpdEmulator;		/* was ppd_emul_t */

  typedef struct {		// *** sRGB Color Profiles ***
    guint8 id;			// Object id

    GString *resolution;	// Resolution or "-" 
    GString *media_type;	// Media type of "-"
    float density;		// Ink density to use 
    float gamma;		// Gamma correction to use 
    float matrix[3][3];		// Transform matrix 
  } PpdProfile;			/* was ppd_profile_t */

  typedef struct {		// *** Files ***
    guint8 id;			// Object id

    int language_level;		// Language level of device 
    gboolean color_device;	// 1 = color device, 0 = grayscale 
    gboolean variable_sizes;	// 1 = supports variable sizes,0 = doesn't 
    gboolean accurate_screens;	// 1 = supports accurate screens, 0 = not 
    gboolean contone_only;	// 1 = continuous tone only, 0 = not 
    int landscape;		// -90 or 90 
    int model_number;		// Device-specific model number 
    gboolean manual_copies;	// 1 = Copies done manually, 0 = hardware 
    PpdCSType colorspace;	// Default colorspace 
    GSList *patches;		// GSList of char*'s Patch commands to be sent
    // to printer 
    GSList *emulations;		// GSList of PpdEmulator's; Emulations and the
    // code to invoke them 
    GString *jcl_begin;		// Start JCL commands 
    GString *jcl_ps;		// Enter PostScript interpreter 
    GString *jcl_end;		// End JCL commands 
    GString *lang_encoding;	// Language encoding 
    GString *lang_version;	// Language version (English, Spanish, etc.) 
    GString *modelname;		// Model name (general) 
    GString *ttrasterizer;	// Truetype rasterizer 
    GString *manufacturer;	// Manufacturer name 
    GString *product;		// Product name (from PS RIP/interpreter) 
    GString *nickname;		// Nickname (specific) 
    GString *shortnickname;	// Short version of nickname

    GSList *groups;		// GSList of PpdGroup objects
    GSList *sizes;		// GSList of PpdSize objects      

    float custom_min[2];	// Minimum variable page size 
    float custom_max[2];	// Maximum variable page size 
    float custom_margins[4];	// Margins around page

    GSList *consts;		// GSList of PpdConstraints
    GSList *fonts;		// GSList of char* 's      
    GSList *profiles;		// GSList of PpdProfiles
    GSList *filters;		// GSList of char*'s
  } PpdFile;			/* used to be: ppd_file_t */



  /* new file prototypes (should we call these constructors?) */
  PpdFile *ppd_file_new(const char *filename);
  PpdFile *ppd_file_new_from_filep(FILE * fp);
  PpdFile *ppd_file_new_from_fd(int fd);


  /* new destructor prototype */
  void ppd_file_free(PpdFile * ppd);

  gboolean ppd_emit_to_file(PpdFile * ppd, FILE * fp, PpdSectionOrder section);
  gboolean ppd_emit_to_fd(PpdFile * ppd, int fd, PpdSectionOrder section);

  int ppd_get_num_conflicts(PpdFile * ppd);

  gboolean ppd_check_option_is_marked(PpdFile * ppd, const char *keyword,
				      const char *option);
  void ppd_mark_defaults(PpdFile * ppd);
  gint ppd_mark_option(PpdFile * ppd, const char *keyword, const char *option);

  PpdChoice *ppd_find_choice(PpdOption * o, const char *option);
  PpdChoice *ppd_find_marked_choice(PpdFile * ppd, const char *keyword);

  PpdOption *ppd_find_option_by_keyword(PpdFile * ppd, const char *keyword);

  float ppd_get_page_length(PpdFile * ppd, const char *name);
  PpdSize *ppd_get_page_size(PpdFile * ppd, const char *name);
  float ppd_get_page_width(PpdFile * ppd, const char *name);

  void ppd_debug_dump_ppd(PpdFile * ppd);

  /* Constructors for the various classes */
  PpdChoice *ppd_choice_new(PpdOption * option, const char *choice,
			    const char *text);
  PpdOption *ppd_option_new(PpdGroup * group, const char *name);
  PpdGroup *ppd_group_new(void);
  PpdConstraint *ppd_constraint_new(void);
  PpdSize *ppd_size_new(const char *name);
  PpdEmulator *ppd_emulator_new(void);
  PpdProfile *ppd_profile_new(void);

// C++ magic...
#ifdef __cplusplus
}
#endif				// __cplusplus
#endif				// !__PPD_H__
// End of ppd.h
