/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef IANIMATOR_H
#define IANIMATOR_H


#include "iobject.h"


#include "iarray.h"
#include "imath.h"
#include "iimage.h"

class iAnimatorEventObserver;
class iAnimatorScript;
class iStereoImageArray;

class vtkImageData;
class vtkPolyData;


struct iAnimatorState 
{
	int mode;
	int nframes;
	int nBlendedFrames;
	int nTransitionFrames;
	float dphi;
	float dtheta;
	float dscale;
	float droll;
	float slideSpeed;
	float flybySpeed;
	double xsecPos;
	bool ifBoundingBox;
	bool ifColorBars;
	bool ifTimeLabel;
	int currec;
	int cameraProjection;
	double cameraPosition[3];
	double cameraFocalPoint[3];
	double cameraViewUp[3];
	double cameraParallelScale;
	double cameraClippingRange[2];
};


class iAnimator : public iObject
{
	
	friend class iAnimatorScript;

	IPOINTER_AS_PART(ViewModule);

public:
	
	vtkTypeMacro(iAnimator,iObject);
	static iAnimator* New(iViewModule *vm = 0);
	static const iObjectType& Type();
		
	IOBJECT_DECLARE_GETSET(Style,mState.mode,int);
	//virtual void SetStyle(int ma);
	//inline int GetStyle() const { return mState.mode; }

	IOBJECT_DECLARE_GETSET(NumberOfFrames,mState.nframes,int);
	//virtual void SetNumberOfFrames(int na);
	//inline int GetNumberOfFrames() const { return mState.nframes; }

	IOBJECT_DECLARE_GETSET(Phi,mState.dphi,float);
	//virtual void SetPhi(float va);
	//inline float GetPhi() const { return mState.dphi; }

	IOBJECT_DECLARE_GETSET(Theta,mState.dtheta,float);
	//virtual void SetTheta(float va);
	//inline float GetTheta() const { return mState.dtheta; }

	IOBJECT_DECLARE_GETSET(Zoom,mState.dscale,float);
	//virtual void SetZoom(float va);
	//inline float GetZoom() const { return mState.dscale; }

	IOBJECT_DECLARE_GETSET(Roll,mState.droll,float);
	//virtual void SetRoll(float va);
	//inline float GetRoll() const { return mState.droll; }

	IOBJECT_DECLARE_GETSET(FlybySpeed,mState.flybySpeed,float);
	//virtual void SetFlybySpeed(float va);
	//inline float GetFlybySpeed() const { return mState.flybySpeed; }

	IOBJECT_DECLARE_GETSET(NumberOfBlendedFrames,mState.nBlendedFrames,int);
	//virtual void SetNumberOfBlendedFrames(int na);
	//inline int GetNumberOfBlendedFrames() const { return mState.nBlendedFrames; }

	IOBJECT_DECLARE_GETSET(NumberOfTransitionFrames,mState.nTransitionFrames,int);
	//virtual void SetNumberOfTransitionFrames(int na);
	//inline int GetNumberOfTransitionFrames() const { return mState.nTransitionFrames; }

	IOBJECT_DECLARE_GETSET1(DebugFlag,int);
	//virtual void SetDebugFlag(int s);
	//inline int GetDebugFlag() const { retun mDebugFlag; }
	static const iObjectKey& KeyDebugging();

	bool SetScriptFile(const iString &fname);
	IOBJECT_DECLARE_GET1(ScriptFile,const iString&);
	//virtual void SetScriptFileName(const iString &s);
	//inline const String& GetScriptFile() const { return mScriptFile; }
	static const iObjectKey& KeyLoadScriptFile();

	bool SetTitlePageFile(const iString &fname);
	IOBJECT_DECLARE_GET1(TitlePageFile,const iString&);
	//inline const iString& GetTitlePageFile() const { return mTitlePageFile; }

	bool SetLogoFile(const iString &fname);
	IOBJECT_DECLARE_GET1(LogoFile,const iString&);
	//inline const iString& GetLogoFile() const { return mLogoFile; }

	IOBJECT_DECLARE_GETSET(NumberOfTitlePageFrames,mTitlePageNumFrames,int);
	//virtual void SetNumberOfTitlePageFrames(int n);
	//inline int GetNumberOfTitlePageFrames() const { return mTitlePageNumFrames; }
	
	IOBJECT_DECLARE_GETSET(NumberOfTitlePageBlendedFrames,mTitlePageNumBlendedFrames,int);
	//virtual void SetNumberOfTitlePageBlendedFrames(int n);
	//inline int GetNumberOfTitlePageBlendedFrames() const { return mTitlePageNumBlendedFrames; }

	IOBJECT_DECLARE_GETSET1(LogoLocation,int);
	//virtual void SetLogoLocation(int n);
	//inline int GetLogoLocation() const { return mLogoLocation; }

	IOBJECT_DECLARE_GETSET1(LogoOpacity,float);
	//virtual void SetLogoOpacity(float v);
	//inline float GetLogoOpacity() const { return mLogoOpacity; }

	inline const iImage& GetTitlePageImage() const { return mTitlePageImage; }
	static const iObjectKey& KeyTitlePageImage();
	inline const iImage& GetLogoImage() const { return mLogoImage; }
	static const iObjectKey& KeyLogoImage();

	virtual void SetCrossSectionSpeed(float va);
	inline float GetCrossSectionSpeed() const { return (float)fabs(mState.slideSpeed); }
	static const iObjectKey& KeyCrossSectionSpeed();

	inline bool UsingScript() const { return mUsingScript; }

	//
	//  Animator controls
	//
	void Animate();
	virtual void Start();
	virtual void Stop();
	virtual bool Continue();

	void AddFollower(iAnimator *f);
	void RemoveFollower(iAnimator *f);
	void RemoveAllFollowers();

	inline iAnimatorEventObserver* GetObserver() const { return mAnimatorObserver; }
	inline iAnimatorScript* GetScript() const { return mAnimatorScript; }
	inline void GetInfo(bool &nr, int &cr, int &cf) const { nr = mNewRec; cr = mCurRec; cf = mCurFrame; }
	virtual void CopyState(iAnimator *other);

protected:

	virtual ~iAnimator();

	virtual void PackStateBody(iString &s) const;
	virtual void UnPackStateBody(const iString &s);

private:
	
	iAnimator(iViewModule *vm);

	void SetLeader(iAnimator *l);

	bool Frame(bool dumpImage = true);
	void DumpImages(iStereoImageArray &images);

	void ResetCurrentFile();
	void SaveState();
	void RestoreState(bool with_camera);

	bool mStarted, mStartedRender, mUsingScript;
	unsigned long mSeed;
	int mDebugFlag;

	iString mScriptFile;

	iAnimatorState mState;
	iAnimatorState mState2;

	float mRandStep;
	bool mNewRec;
	int mPrevRec, mCurRec, mTotFrame, mCurFrame;

	iArray<iStereoImageArray*> mBlenderBase;

	iString mTitlePageFile, mLogoFile;
	iImage mTitlePageImage, mLogoImage;
	int mTitlePageNumBlendedFrames, mTitlePageNumFrames, mLogoLocation;
	float mLogoOpacity;

	iAnimatorScript *mAnimatorScript;
	iAnimatorEventObserver *mAnimatorObserver;

	iSearchableArray<iAnimator*> mFollowers;
	iAnimator *mLeader;

	//
	//  Internal (work) variables
	//
	struct AnimatorData
	{
		double pos[3];
		float r, dphl0, dthl0, ramp;
		double xc1[3], xc2[3], x[3];
		float v[3], t, dt0, dt;
	} wData;
};

#endif // IANIMATOR_H

