///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __SCRIPT_ENGINE_H
#define __SCRIPT_ENGINE_H

#include <scripting/Scripting.h>

namespace Scripting {

/// \brief Adds the initXXX() function of a plugin to the internal list.
/// \param pluginName The identifier of the plugin to register.
/// \param initFunc The initXXX() function to be passed to the Python interpreter.
/// Use the OVITO_REGISTER_PLUGIN_PYTHON_INTERFACE macro to call this method on application startup.
struct SCRIPTING_DLLEXPORT PythonPluginRegistration
{
	const char* _pluginName;
	void (*_initFunc)();
	PythonPluginRegistration* next;
	PythonPluginRegistration(const char* pluginName, void (*initFunc)()) : _pluginName(pluginName), _initFunc(initFunc) {
		next = instances;
		instances = this;
	}

	/// The initXXX() functions for each of the registered plugins.
	static PythonPluginRegistration* instances;
};

/// This macro must be used by every plugin that exposes a Python
/// scripting interface using Boost.Python.
#define OVITO_REGISTER_PLUGIN_PYTHON_INTERFACE(pluginName) \
	static Scripting::PythonPluginRegistration __scripting_unused_variable##pluginName( #pluginName , init##pluginName);

/**
 * \brief Encapsulates the Python interpreter.
 *
 * \author Alexander Stukowski
 */
class SCRIPTING_DLLEXPORT ScriptEngine : public QObject
{
	Q_OBJECT

public:

	/// \brief Constructor.
	/// \param parent The owner of the script engine object.
	/// \throw Exception if the Python interpreter could not be initialized.
	ScriptEngine(QObject* parent = NULL);

	/// \brief Returns a reference to the global master engine.
	/// \return The master script engine that runs scripts in the global scope.
	/// \throw Exception if the Python interpreter could not be initialized.
	static ScriptEngine& master();

public Q_SLOTS:

	/// \brief Executes a Python script.
	/// \param script The script source code.
	/// \return The exit code returned by the python script.
	/// \throw Exception on error.
	int executeScript(const QByteArray& script);

	/// \brief Executes a Python script.
	/// \param script The script source code.
	/// \return The exit code returned by the python script.
	/// \throw Exception on error.
	int executeScript(const QString& script) { return executeScript(script.toAscii()); }

	/// \brief Executes a Python script file.
	/// \param scriptFile The script file path.
	/// \return The exit code returned by the python script.
	/// \throw Exception on error.
	int executeScriptFile(const QString& scriptFile);

	/// \brief Writes the given string to stdout of the host program.
	/// \param outputString The string to send to stdout.
	///
	/// This slot can be conntected to the engine's scriptOutput() signal
	/// to send the Python script's output messages to the console.
	void consoleOutput(const QString& outputString) {
		// Forward script output to console.
		std::cout << outputString.toLocal8Bit().constData();
	}

	/// \brief Writes the given string to stderr of the host program.
	/// \param outputString The string to send to stderr.
	///
	/// This slot can be conntected to the engine's scriptError() signal
	/// to send the Python script's error messages to the console.
	void consoleError(const QString& errorString) {
		// Forward script output to console.
		std::cerr << errorString.toLocal8Bit().constData();
	}

Q_SIGNALS:

	/// This signal is emmited when the Python script has generated some screen output.
	void scriptOutput(const QString& outputString);

	/// This is emmited when the Python script has generated some error output.
	void scriptError(const QString& errorString);

protected:

	/// \brief Private constructor used to construct the master engine.
	ScriptEngine(bool) : failedToInitialize(true) {}

	/// \brief Initializes the Python interpreter.
	/// \throw Exception on error.
	void initInterpreter();

	/// \brief This will register the output redirector objects
	///        to catch the Python output to stdout/stderr.
	void setupOutputRedirector();

	/// \brief Register the Python classes exposed by a plugin with the Python interpreter.
	/// \param plugin The plugin to be registered.
	void registerPluginWithInterpreter(Plugin* plugin, QSet<Plugin*>& listOfAlreadyLoadedPlugins);
	
	/// \brief Handles a call to sys.exit() in the Python interpreter.
	/// \return The exit code.
	int handleSystemExit();

	/// \brief Sends some text to the output sink.
	void writeOutput(const QString& str, bool isError = false) {
		if(!isError)
			scriptOutput(str);
		else
			scriptError(str);
	}

	// Redirects the interpreter's output to the output window.
	class InterpreterOutputRedirector {
	public:
		InterpreterOutputRedirector(ScriptEngine* p, bool e) : engine(p), isError(e) {}
		void write(const QString& str) { if(engine) engine->writeOutput(str, isError); }
	private:
		QPointer<ScriptEngine> engine;
		bool isError;
	};

	/// Indicates that there was an error when initializing the Python interpreter.
	bool failedToInitialize;

	/// The namespace (scope) of the current script.
	boost::python::object main_namespace;

	/// Array of char* pointers to the registered module names.
	static QVector< shared_array<char> > moduleNames;

	/// The master script engine that runs in the global scope.
	static ScriptEngine* masterEngine;

	/// This cleanup handler is used to delete the master engine when the program ends.
	static QObjectCleanupHandler masterEngineCleanupHandler;

};

};

#endif // __SCRIPT_ENGINE_H
