/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

#include "EndpointReference.h"

using namespace CimClassNamespace;

SelectorSet::SelectorSet():
Serializable("SelectorSet", "http://schemas.dmtf.org/wbem/wsman/1/wsman.xsd", "wsman"),
DeSerializable("SelectorSet", "http://schemas.dmtf.org/wbem/wsman/1/wsman.xsd"){}

SelectorSet::~SelectorSet(){}

void SelectorSet::SerializeMembers(XMLElement& elem) const
{
	for(NameValuePairs::const_iterator it = Selectors.begin() ; it != Selectors.end(); it++)
	{
		XMLElement tmp = elem.CreateChildNode("Selector", nsUri, prefix, &it->second);
		tmp.AddAttribValue("Name", it->first);
	}
}

void SelectorSet::SetValue(const XMLElement& elem)
{
	if(elem.GetNodeName() == "Selector")
	{
		Selectors[elem.GetAttribValue("Name")] = elem.ToString();
	}
}


ReferenceParameters::ReferenceParameters():
Serializable("ReferenceParameters", "http://schemas.xmlsoap.org/ws/2004/08/addressing", "wsa"),
DeSerializable("ReferenceParameters", "http://schemas.xmlsoap.org/ws/2004/08/addressing"){}

ReferenceParameters::~ReferenceParameters(){}

void ReferenceParameters::SerializeMembers(XMLElement& elem) const
{
	string wsman = "wsman";
	string wsmanNs = "http://schemas.dmtf.org/wbem/wsman/1/wsman.xsd";
	elem.AddNSDefinition(wsmanNs, &wsman);
	elem.AppendLeaf("ResourceUri", uri, wsmanNs, wsman);
	elem.AppendNode("SelectorSet", Selectors, wsmanNs, wsman);
}

NameValuePairs ReferenceParameters::GetSelectors() const
{
	return Selectors.Selectors;
}

void ReferenceParameters::SetSelectors(const NameValuePairs &sel)
{
	Selectors.Selectors.clear();
	Selectors.Selectors = sel;
}

void ReferenceParameters::SetValue(const XMLElement& elem)
{
	if(elem.GetNodeName() == "ResourceURI")
	{
		uri = elem.GetTextValue();
	}
	else if(elem.GetNodeName() == "SelectorSet")
	{
		Selectors.Deserialize(elem);
	}
}

void ReferenceParameters::SetUri(const string& newUri)
{
	uri = newUri;
}

string ReferenceParameters::GetUri() const
{
	return uri;
}

EndpointReference::EndpointReference(const string& name,
									 const string& ns,
									 const string& pref):
Serializable(name, ns, pref),
DeSerializable(name, ns), eprName(name){add = "default";}

EndpointReference::~EndpointReference(){}

string EndpointReference::GetEPRName() const
{
	return eprName;
}

void EndpointReference::SetName(const string &name)
{
	eprName = name;
}

void EndpointReference::SetAddress(const string &newAdd)
{
	add = newAdd;
}

string EndpointReference::GetAddress() const
{
	return add;
}

void EndpointReference::SerializeMembers(XMLElement& elem) const
{
	string wsa = "wsa";
	string wsaNs = "http://schemas.xmlsoap.org/ws/2004/08/addressing";
	elem.AddNSDefinition(wsaNs, &wsa);
	elem.AppendLeaf("Address", add, wsaNs, wsa);
	elem.AppendNode("ReferenceParameters", ref, wsaNs, wsa);
}

void EndpointReference::SetValue(const XMLElement& elem)
{
	if(elem.GetNodeName() == "Address")
	{
		add = elem.GetTextValue();
	}
	else if(elem.GetNodeName() == "ReferenceParameters")
	{
		ref.Deserialize(elem);
	}
}

NameValuePairs EndpointReference::GetSelectors() const
{
	return ref.GetSelectors();
}

string EndpointReference::GetResourceURI() const
{
	return ref.GetUri();
}

void EndpointReference::SetResourceUri(const string& uri)
{
	ref.SetUri(uri);
}

void EndpointReference::SetSelectors(const NameValuePairs &sel)
{
	ref.SetSelectors(sel);
}

bool EndpointReference::CompareResourceURI(const string &uri) const
{
	return (ref.GetUri().compare(uri) == 0);
}

ResourceCreated::ResourceCreated():
EndpointReference("ResourceCreated", "http://schemas.xmlsoap.org/ws/2004/09/transfer", "wxf"){}

ResourceCreated::~ResourceCreated(){}
