'''
blindSqli.py

Copyright 2006 Andres Riancho

This file is part of w3af, w3af.sourceforge.net .

w3af is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation version 2 of the License.

w3af is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with w3af; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

'''
import core.controllers.outputManager as om
import core.data.kb.knowledgeBase as kb

from core.controllers.basePlugin.baseAuditPlugin import baseAuditPlugin
from core.controllers.sql_tools.blind_sqli_response_diff import \
    blind_sqli_response_diff
from core.controllers.sql_tools.blind_sqli_time_delay import \
    blind_sqli_time_delay

from core.data.options.option import option
from core.data.options.optionList import optionList
from core.data.fuzzer.fuzzer import createMutants



class blindSqli(baseAuditPlugin):
    '''
    Identify blind SQL injection vulnerabilities.
    
    @author: Andres Riancho ( andres.riancho@gmail.com )
    '''

    def __init__(self):
        baseAuditPlugin.__init__(self)
        
        # User configured variables
        self._eq_limit = 0.9

    def audit(self, freq):
        '''
        Tests an URL for blind SQL injection vulnerabilities.
        
        @param freq: A fuzzableRequest
        '''
        om.out.debug('blindSqli plugin is testing: ' + freq.getURL())
        
        #
        #    Setup blind SQL injection detector objects
        #
        self._bsqli_response_diff = blind_sqli_response_diff(self._uri_opener)
        bsqli_resp_diff = self._bsqli_response_diff
        bsqli_resp_diff.set_eq_limit(self._eq_limit)
        
        self._blind_sqli_time_delay = blind_sqli_time_delay(self._uri_opener)
        bsqli_time_delay = self._blind_sqli_time_delay
        
        method_list = [bsqli_resp_diff, bsqli_time_delay]
        
        #
        #    Use the objects to identify the vulnerabilities
        #
        fake_mutants = createMutants(freq, ['',])
        
        for mutant in fake_mutants:
            
            if self._has_sql_injection( mutant ):
                #
                #    If sqli.py was enabled and already detected a vulnerability
                #    in this parameter, then it makes no sense to test it again
                #    and report a duplicate to the user
                #
                continue
            
            
            for method in method_list:
                found_vuln = method.is_injectable( mutant )

                if found_vuln is not None and \
                self._has_no_bug(freq, varname=found_vuln.getVar()):
                    om.out.vulnerability(found_vuln.getDesc())
                    kb.kb.append(self, 'blindSqli', found_vuln)
                    break
    
    def _has_sql_injection(self, mutant):
        '''
        @return: True if there IS a reported SQL injection for this URL/parameter
                 combination.
        '''
        sql_injection_list = kb.kb.getData('sqli','sqli')
        for sql_injection in sql_injection_list:
            if  sql_injection.getURL() == mutant.getURL() and \
                sql_injection.getVar() == mutant.getVar():
                return True
        
        return False
    
    def getOptions( self ):
        '''
        @return: A list of option objects for this plugin.
        '''
        ol = optionList()
        
        d = 'String equal ratio (0.0 to 1.0)'
        h = 'Two pages are considered equal if they match in more than eq_limit.'
        o = option('eq_limit', self._eq_limit, d, 'float', help=h)
        
        ol.add(o)
        
        return ol

    def setOptions( self, optionsMap ):
        '''
        This method sets all the options that are configured using the user interface 
        generated by the framework using the result of getOptions().
        
        @parameter OptionList: A dictionary with the options for the plugin.
        @return: No value is returned.
        ''' 
        self._eq_limit = optionsMap['eq_limit'].getValue()

    def getPluginDeps( self ):
        '''
        @return: A list with the names of the plugins that should be run before the
        current one.
        '''
        return []

    def getLongDesc( self ):
        '''
        @return: A DETAILED description of the plugin functions and features.
        '''
        return '''
        This plugin finds blind SQL injections using two techniques: time delays
        and true/false response comparison.
        
        Only one configurable parameters exists:
            - eq_limit
        
        '''
