/* startprogop.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2001-2014 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "startprogop.h"
#include "listermode.h"
#include "worker.h"
#include <string>
#include <aguix/kartei.h>
#include <aguix/acontainerbb.h>
#include "execlass.h"
#include "basic_actions.h"
#include "wcfiletype.hh"
#include "avfssupport.hh"
#include "fileentry.hh"
#include "nmspecialsourceext.hh"
#include "fileviewer.hh"
#include "processexitaction.hh"
#include "datei.h"
#include "worker_locale.h"
#include <aguix/fieldlistview.h>
#include <aguix/cyclebutton.h>
#include <aguix/choosebutton.h>
#include <aguix/stringgadget.h>
#include <aguix/button.h>
#include <aguix/textview.h>
#include "wconfig.h"
#include "flattypelist.hh"
#include "argclass.hh"
#include "nwc_path.hh"
#include "pers_string_list.hh"
#include "wpucontext.h"
#include "ownop.h"

const char *StartProgOp::name="StartProgOp";

StartProgOp::StartProgOp() : FunctionProto()
{
  startprogstart=STARTPROGOP_START_NORMAL;

  view_str = "";
  
  global=false;
  inbackground = false;
  dontcd = false;
  hasConfigure = true;

  gui_msg = "";

  m_cmd_sg = nullptr;
  m_cmd_flag_b = nullptr;
}

StartProgOp::~StartProgOp()
{
}

StartProgOp*
StartProgOp::duplicate() const
{
  StartProgOp *ta=new StartProgOp();
  ta->startprogstart=startprogstart;

  ta->view_str = view_str;
  ta->global=global;
  ta->inbackground = inbackground;
  ta->dontcd = dontcd;
  return ta;
}

bool
StartProgOp::isName(const char *str)
{
  if(strcmp(str,name)==0) return true; else return false;
}

const char *
StartProgOp::getName()
{
  return name;
}

int
StartProgOp::run( WPUContext *wpu, ActionMessage *msg )
{
  if(msg->mode!=msg->AM_MODE_DNDACTION) {
    Lister *l1 = msg->getWorker()->getActiveLister();
    if(l1!=NULL) {
      startlister=l1;
      endlister = msg->getWorker()->getOtherLister(startlister);
      startprog( wpu, msg );
    }
  }
  return 0;
}

bool
StartProgOp::save(Datei *fh)
{
  if ( fh == NULL ) return false;
  switch(startprogstart) {
    case STARTPROGOP_START_IN_TERMINAL:
      fh->configPutPair( "start", "terminal" );
      break;
    case STARTPROGOP_START_IN_TERMINAL_AND_WAIT4KEY:
      fh->configPutPair( "start", "terminalwait" );
      break;
    case STARTPROGOP_SHOW_OUTPUT:
      fh->configPutPair( "start", "showoutput" );
      break;
    case STARTPROGOP_SHOW_OUTPUT_INT:
      fh->configPutPair( "start", "showoutputint" );
      break;
    default:
      fh->configPutPair( "start", "normal" );
      break;
  }
  fh->configPutPairString( "viewstr", view_str.c_str() );
  fh->configPutPairBool( "global", global );
  fh->configPutPairBool( "inbackground", inbackground );
  fh->configPutPairBool( "dontcd", dontcd );
  return true;
}

const char *
StartProgOp::getDescription()
{
  return catalog.getLocaleCom(32);
}

/* replace backslash with double backslash and backslash { and } */
static std::string protectForParsing( const char *str1 )
{
    std::string res;

    if ( str1 != NULL ) {
        for ( ; *str1 != '\0'; str1++ ) {
            switch ( *str1 ) {
                case '\\':
                    res += "\\\\";
                    break;
                case '{':
                    res += "\\{";
                    break;
                case '}':
                    res += "\\}";
                    break;
                default:
                    res += *str1;
                    break;
            }
        }
    }

    return res;
}

int
StartProgOp::startprog( WPUContext *wpu, ActionMessage *am )
{
  ListerMode *lm1 = NULL;

  char *tmpname, *tmpoutput, *tstr;
  std::string string1;
  bool useint;
  bool removeTMPFiles;
  std::string fullname;

  if ( startlister == NULL ) return 1;
  lm1 = startlister->getActiveMode();
  if ( lm1 == NULL ) return 1;

  std::list< NM_specialsourceExt > filelist;

  lm1->getSelFiles( filelist, ListerMode::LM_GETFILES_ONLYACTIVE );

  const FileEntry *tfe;
  if ( am->mode == am->AM_MODE_SPECIAL ) {
    // Copy of FE is not necessary because it's already duplicated in ActionMessage
    tfe = am->getFE();
  } else if ( ! filelist.empty() &&
              filelist.begin()->entry() != NULL ) {
    tfe = filelist.begin()->entry();
  } else {
    tfe = NULL;
  }
  if ( tfe != NULL ) {
      std::string cwd = lm1->getCurrentDirectory();

      default_file = tfe->name;
      fullname = tfe->fullname;

      if ( ! cwd.empty() ) {
          std::string name_string = NWC::Path::get_extended_basename( cwd,
                                                                      tfe->fullname );
          if ( ! name_string.empty() ) {
              default_file = name_string;
          }
      }
  } else {
    default_file = "";
  }

  std::unique_ptr< StartSettings > m1;
  m1 = showGUI();
  if ( m1->_startmode == StartSettings::STARTPROG ) {
    tmpname = Datei::createTMPName();
    tmpoutput = Datei::createTMPName();

    if ( tmpname != NULL && tmpoutput != NULL ) {
      removeTMPFiles = true;

      std::string tmpoutput_lnk;

      RefCount< GenericCallbackArg< void, int > > pea( NULL );
      
      if ( m1->_start == STARTPROGOP_SHOW_OUTPUT_INT ) {
          tmpoutput_lnk = Datei::createTMPHardlink( tmpoutput );

          wpu->ref();

          pea = new ProcessExitAction( true,
                                       tmpoutput_lnk,
                                       am->getWorker(), wpu );
      } else {
          wpu->ref();

          pea = new ProcessExitAction( false,
                                       "",
                                       am->getWorker(), wpu );
      }

      Datei fp;
      if ( fp.open( tmpname, "w" ) == 0 ) {
        fp.putString( "#! /bin/sh\n" );
        if ( ! lm1->getCurrentDirectory().empty() ) {
            if ( ( worker_islocal( lm1->getCurrentDirectory().c_str() ) == 1 ) &&
                 ( m1->_dontcd == false ) ) {
                tstr = AGUIX_catTrustedAndUnTrusted( "cd ", lm1->getCurrentDirectory().c_str() );
                fp.putString( tstr );
                _freesafe( tstr );
                fp.putString( "\n" );
            }
        }
        if ( m1->_global == false ) fp.putString( "./" );

        std::string res_str;
        if ( wpu->parse( m1->_command.c_str(),
                         res_str,
                         EXE_STRING_LEN - strlen( tmpoutput ) - 3 ) == WPUContext::PARSE_SUCCESS ) {
            string1 = Worker_secureCommandForShell( res_str.c_str() );
        }

        if ( string1.length() > 0 ) {
          if ( m1->_start == STARTPROGOP_SHOW_OUTPUT ||
               m1->_start == STARTPROGOP_SHOW_OUTPUT_INT ) {
            string1 += " >";
            string1 += tmpoutput;
          }
          fp.putString( string1.c_str() );
          fp.putString( "\n" );
        }

        std::string exestr;

        if ( ( m1->_start == STARTPROGOP_START_IN_TERMINAL_AND_WAIT4KEY )||
             ( m1->_start == STARTPROGOP_START_IN_TERMINAL ) ) {
          /* in terminal starten */
          if ( m1->_start == STARTPROGOP_START_IN_TERMINAL_AND_WAIT4KEY ) {
            fp.putStringExt( "echo %s\n", catalog.getLocale( 201 ) );
            fp.putString( "read x\n" );
          }
          exestr = AGUIXUtils::formatStringToString( wconfig->getTerminalStr(),
                                                     tmpname );
        } else {
            exestr = "/bin/sh ";
            exestr += tmpname;
        }
        if ( m1->_start == STARTPROGOP_SHOW_OUTPUT ) {
          useint = true;
          if ( m1->_view_str.length() > 2 )
            if ( strstr( m1->_view_str.c_str(), "%s" ) != NULL ) useint = false;
          if ( useint == true ) fp.putStringExt( OUTPUT_BIN, tmpoutput );
          else fp.putStringExt( m1->_view_str.c_str(), tmpoutput );
          fp.putString( "\n" );
        }
        fp.close();

        if ( fp.errors() == 0 ) {
          worker_chmod( tmpname, 0700 );
          removeTMPFiles = false;
          am->getWorker()->runCommand( exestr.c_str(), tmpname, tmpoutput, m1->_inbackground, pea );
        } else {
          Requester req( Worker::getAGUIX() );
          char *textstr = (char*)_allocsafe( strlen( catalog.getLocale( 647 ) ) + strlen( tmpname ) + 1 );

          sprintf( textstr, catalog.getLocale( 647 ), tmpname );
          std::string button = catalog.getLocale( 11 );
          req.request( catalog.getLocale( 347 ), textstr, button.c_str() );
          _freesafe( textstr );
        }
      }
      if ( removeTMPFiles == true ) {
        worker_unlink( tmpname );
        worker_unlink( tmpoutput );

        if ( pea.getVal() != NULL ) {
            pea->callback( 0 );
        }
      }
    } else {
      Requester req( Worker::getAGUIX() );
      char *textstr = (char*)_allocsafe( strlen( catalog.getLocale( 647 ) ) + strlen( "" ) + 1 );
      
      sprintf( textstr, catalog.getLocale( 647 ), "" );
      std::string button = catalog.getLocale( 11 );
      req.request( catalog.getLocale( 347 ), textstr, button.c_str() );
      _freesafe( textstr );
    }
    
    if ( tmpname != NULL ) _freesafe( tmpname );
    if ( tmpoutput != NULL ) _freesafe( tmpoutput );
  } else if ( m1->_startmode == StartSettings::HANDLE_TYPE ) {
    WCFiletype *ft = m1->_type;
    if ( ( ft != NULL ) && ( tfe != NULL ) ) {
      List *l = ft->getDoubleClickActions();
      if ( l != NULL ) {
	ActionMessage amsg( am->getWorker() );
	
	amsg.mode = amsg.AM_MODE_SPECIAL;
	
	amsg.setFE( tfe );
	amsg.filetype = ft;
        amsg.m_action_descr = RefCount<ActionDescr>( new DoubleClickAction::DoubleClickActionDescr() );
	am->getWorker()->interpret( l, &amsg );
      }
    }
  } else if ( m1->_startmode == StartSettings::HANDLE_ARCHIVE ) {
    std::string dir;

    if ( tfe != NULL ) {
      dir = tfe->fullname;
      dir += "#";
      dir += m1->_archive;

      std::list< RefCount< ArgClass > > args;

      args.push_back( new StringArg( dir ) );
      lm1->runCommand( "enter_dir", args );
    }
  }

  if ( m1->_startmode != StartSettings::CANCEL ) {
      am->getWorker()->storePathPersIfInProgress( fullname );
  }

  return 0;
}

int
StartProgOp::configure()
{
  return doconfigure();
}

std::unique_ptr< StartProgOp::SettingsWidgets > StartProgOp::buildW( AWindow *win )
{
  SettingsWidgets *m1;

  AGUIX *aguix = Worker::getAGUIX();
  StringGadget *sg;
  CycleButton *cyb;
  ChooseButton *gcb, *ibcb, *dcdcb;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cincwnr = cincw +
                      AContainer::ACONT_NORESIZE;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;
  
  AContainer *ac1 = win->setContainer( new AContainerBB( win, 1, 6 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  ac1->add( new Text( aguix, 0, 0, catalog.getLocale( 638 ) ), 0, 0, cincwnr );

  AContainer *ac1_1 = ac1->add( new AContainer( win, 2, 1 ), 0, 1 );
  ac1_1->setMinSpace( 5 );
  ac1_1->setMaxSpace( 5 );
  ac1_1->setBorderWidth( 0 );

  ac1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 331 ) ), 0, 0, cfix );
  cyb = (CycleButton*)ac1_1->add( new CycleButton( aguix, 0, 0, 100, 0 ), 1, 0, cincw );
  cyb->addOption(catalog.getLocale(332));
  cyb->addOption(catalog.getLocale(333));
  cyb->addOption(catalog.getLocale(334));
  cyb->addOption(catalog.getLocale(335));
  cyb->addOption( catalog.getLocale( 911 ) );
  cyb->resize(cyb->getMaxSize(),cyb->getHeight());
  switch(startprogstart) {
    case STARTPROGOP_START_IN_TERMINAL:
      cyb->setOption(1);
      break;
    case STARTPROGOP_START_IN_TERMINAL_AND_WAIT4KEY:
      cyb->setOption(2);
      break;
    case STARTPROGOP_SHOW_OUTPUT:
      cyb->setOption(3);
      break;
    case STARTPROGOP_SHOW_OUTPUT_INT:
      cyb->setOption( 4 );
      break;
    default:
      cyb->setOption(0);
      break;
  }
  ac1_1->readLimits();
  
  AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 2 );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( 5 );
  ac1_2->setBorderWidth( 0 );

  ac1_2->add( new Text( aguix, 0, 0, catalog.getLocale( 337 ) ), 0, 0, cfix );
  sg = (StringGadget*)ac1_2->add( new StringGadget( aguix, 0, 0, 100, view_str.c_str(), 0 ), 1, 0, cincw );

  gcb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, ( global == true ) ? 1 : 0,
						   catalog.getLocale( 343 ), LABEL_RIGHT, 0 ), 0, 3, cincwnr );
  ibcb = (ChooseButton*)ac1->add( new ChooseButton( aguix,
                                                    0,
                                                    0,
                                                    ( inbackground == true ) ? 1 : 0,
                                                    catalog.getLocale( 435 ),
                                                    LABEL_RIGHT,
                                                    0 ), 0, 4, cincwnr );

  dcdcb = (ChooseButton*)ac1->add( new ChooseButton( aguix,
						     0,
						     0,
						     ( dontcd == true ) ? 1 : 0,
						     catalog.getLocale( 617 ),
						     LABEL_RIGHT,
						     0 ), 0, 5, cincwnr );

  win->contMaximize( true );
  win->show();
  
  m1 = new SettingsWidgets( sg, cyb, gcb, ibcb, dcdcb );

  return std::unique_ptr< SettingsWidgets >( m1 );
}

int StartProgOp::doconfigure()
{
  AGUIX *aguix = Worker::getAGUIX();
  Button *okb,*cb;
  AWindow *win;
  AGMessage *msg;
  int endmode=-1;
  char *tstr;
  int i;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;
  const int cmin = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW;
  
  tstr=(char*)_allocsafe(strlen(catalog.getLocale(293))+strlen(catalog.getLocaleCom(32))+1);
  sprintf(tstr,catalog.getLocale(293),catalog.getLocaleCom(32));
  win = new AWindow( aguix, 10, 10, 10, 10, tstr, AWindow::AWINDOW_DIALOG );
  win->create();
  _freesafe(tstr);

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 2 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  AWindow *subwin1 = new AWindow( aguix, 10, 10, 10, 10, "" );
  ac1->add( subwin1, 0, 0, cmin );

  std::unique_ptr< SettingsWidgets > m1 = buildW( subwin1 );
  ac1->readLimits();

  AContainer *ac1_3 = ac1->add( new AContainer( win, 2, 1 ), 0, 1 );
  ac1_3->setMinSpace( 5 );
  ac1_3->setMaxSpace( -1 );
  ac1_3->setBorderWidth( 0 );
  okb =(Button*)ac1_3->add( new Button( aguix,
					0,
					0,
					catalog.getLocale( 11 ),
					0 ), 0, 0, cfix );
  cb = (Button*)ac1_3->add( new Button( aguix,
					0,
					0,
					catalog.getLocale( 8 ),
					0 ), 1, 0, cfix );

  okb->takeFocus();
  win->setDoTabCycling( true );
  win->contMaximize( true );
  win->show();

  for(;endmode==-1;) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==win->getWindow()) endmode=1;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==okb) {
	    if ( m1->validate() == 1 ) endmode = 0;
          } else if(msg->button.button==cb) endmode=1;
          break;
	case AG_STRINGGADGET_DEACTIVATE:
          if(msg->stringgadget.sg==m1->_sg) {
	    m1->validate();
          }
	  break;
        case AG_KEYPRESSED:
          if(win->isParent(msg->key.window,false)==true) {
            switch(msg->key.key) {
              case XK_1:
                i=m1->_cyb->getSelectedOption()+1;
                if(i>=4) i=0;
                m1->_cyb->setOption(i);
                break;
              case XK_2:
                m1->_gcb->setState((m1->_gcb->getState()==true)?false:true);
                break;
              case XK_3:
                m1->_ibcb->setState( ( m1->_ibcb->getState() == true ) ? false : true );
                break;
              case XK_Return:
                if ( cb->getHasFocus() == false ) {
		  if ( m1->validate() == 1 ) endmode = 0;
                }
                break;
              case XK_Escape:
                endmode=1;
                break;
            }
          }
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }
  
  if(endmode==0) {
    // ok
    view_str = m1->_sg->getText();
    switch(m1->_cyb->getSelectedOption()) {
      case 1:
	startprogstart=STARTPROGOP_START_IN_TERMINAL;
	break;
      case 2:
	startprogstart=STARTPROGOP_START_IN_TERMINAL_AND_WAIT4KEY;
	break;
      case 3:
	startprogstart=STARTPROGOP_SHOW_OUTPUT;
	break;
      case 4:
	startprogstart = STARTPROGOP_SHOW_OUTPUT_INT;
	break;
      default:
	startprogstart=STARTPROGOP_START_NORMAL;
	break;
    }
    global = m1->_gcb->getState();
    inbackground = m1->_ibcb->getState();
    dontcd = m1->_dcdcb->getState();
  }
  
  delete win;

  return endmode;
}

void StartProgOp::setStart(startprogstart_t nv)
{
  startprogstart=nv;
}

void StartProgOp::setGlobal(bool nv)
{
  global=nv;
}

void StartProgOp::setRequestFlags(bool nv)
{
}

void StartProgOp::setInBackground( bool nv )
{
  inbackground = nv;
}

void StartProgOp::setDontCD( bool nv )
{
  dontcd = nv;
}

void StartProgOp::setViewStr( std::string nv )
{
  view_str = nv;
}

StartProgOp::SettingsWidgets::SettingsWidgets( StringGadget *sg,
                                               CycleButton *cyb,
                                               ChooseButton *gcb,
                                               ChooseButton *ibcb,
                                               ChooseButton *dcdcb ) : _sg( sg ),
                                                                       _cyb( cyb ),
                                                                       _gcb( gcb ),
                                                                       _ibcb( ibcb ),
    _dcdcb( dcdcb )
{
}

static bool isCorrectViewProg( const char *str )
{
    const char *pos;
    if ( strlen( str ) < 1 ) return true;
    pos = strstr( str, "%s" );
    if ( pos != NULL ) return true;
    return false;
}

int StartProgOp::SettingsWidgets::validate()
{
  const char *textstr;
  char *buttonstr;
  int erg;
  Requester req( Worker::getAGUIX() );

  if ( isCorrectViewProg( _sg->getText() ) == false ) {
    // show requester
    textstr = catalog.getLocale( 312 );
    buttonstr = (char*)_allocsafe( strlen( catalog.getLocale( 313 ) ) + 1 +
				   strlen( catalog.getLocale( 314 ) ) + 1 );
    sprintf( buttonstr, "%s|%s", catalog.getLocale( 313 ),
	     catalog.getLocale( 314 ) );
    erg = req.request( catalog.getLocale( 125 ), textstr, buttonstr );
    _freesafe( buttonstr );
    if ( erg == 0 ) {
      _sg->activate();
    } else {
      _sg->setText( "" );
    }
  } else return 1;
  return 0;
}

std::unique_ptr< StartProgOp::StartSettings > StartProgOp::showGUI()
{
  AGUIX *aguix = Worker::getAGUIX();
  int w, h, sw;
  AWindow *win;
  int row;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cincwnr = cincw +
                      AContainer::ACONT_NORESIZE;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;
  const int cmin = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW;
  int i;
  std::string str1;

  if ( gui_msg.length() < 1 )
    gui_msg = catalog.getLocale( 645 );

  win = new AWindow( aguix, 10, 10, 10, 10, catalog.getLocaleCom( 32 ) );
  win->create();

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 3 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );
  
  ac1->add( new Text( aguix, 0, 0, gui_msg.c_str() ), 0, 0, cincwnr );

  Kartei *k1 = new Kartei( aguix, 10, 10, 10, 10, "" );
  ac1->add( k1, 0, 1, cmin );
  k1->create();

  AWindow *subwin11 = new AWindow( aguix, 0, 0, 10, 10, "" );
  k1->add( subwin11 );
  subwin11->create();
  
  AContainer *acsw11 = subwin11->setContainer( new AContainer( subwin11, 1, 3 ), true );
  acsw11->setMinSpace( 5 );
  acsw11->setMaxSpace( 5 );
  acsw11->setBorderWidth( 5 );

  RefCount<AFontWidth> lencalc( new AFontWidth( aguix, NULL ) );
  TextStorageString help_ts( catalog.getLocale( 1076 ), lencalc );
  TextView *help_tv = (TextView*)acsw11->add( new TextView( aguix,
                                                            0, 0, 50, 80, "", help_ts ),
                                              0, 0, AContainer::CO_INCW );
  help_tv->setLineWrap( true );
  help_tv->maximizeX( 500 );
  help_tv->maximizeYLines( 10 );
  help_tv->showFrame( false );
  acsw11->readLimits();
  help_tv->show();
  help_tv->setAcceptFocus( false );

  TextView::ColorDef tv_cd = help_tv->getColors();
  tv_cd.setBackground( 0 );
  tv_cd.setTextColor( 1 );
  help_tv->setColors( tv_cd );

  AContainer *acsw11_1 = acsw11->add( new AContainer( subwin11, 2, 2 ), 0, 1 );
  acsw11_1->setMinSpace( 5 );
  acsw11_1->setMaxSpace( 5 );
  acsw11_1->setBorderWidth( 0 );

  std::string cmd_string;
  {
      char *tstr = AGUIX_catQuotedAndUnQuoted( "", default_file.c_str() );
      cmd_string = protectForParsing( tstr );
      _freesafe( tstr );
  }

  acsw11_1->add( new Text( aguix, 0, 0, catalog.getLocale( 639 ) ), 0, 0, cfix );

  AContainer *acsw11_1_1 = acsw11_1->add( new AContainer( subwin11, 2, 1 ), 1, 0 );
  acsw11_1_1->setMinSpace( 0 );
  acsw11_1_1->setMaxSpace( 0 );
  acsw11_1_1->setBorderWidth( 0 );

  m_cmd_sg = acsw11_1_1->addWidget( new StringGadget( aguix, 0,
                                                      0, 100, cmd_string.c_str(), 0 ), 0, 0, cincw );

  m_cmd_flag_b = acsw11_1_1->addWidget( new Button( aguix,
                                                    0,
                                                    0,
                                                    "F",
                                                    0 ), 1, 0, cfix );
  m_cmd_flag_b->setBubbleHelpText( catalog.getLocale( 1131 ) );

  acsw11_1->add( new Text( aguix, 0, 0, catalog.getLocale( 1035 ) ), 0, 1,
                 AContainer::ACONT_MINH +
                 AContainer::ACONT_MINW +
                 AContainer::ACONT_MAXW +
                 AContainer::ACONT_NORESIZE );
  m_completion_lv = acsw11_1->addWidget( new FieldListView( aguix, 0, 0, 100, 75, 0 ),
                                         1, 1,
                                         cmin );
  m_completion_lv->setHBarState( 2 );
  m_completion_lv->setVBarState( 2 );

  AWindow *subwin11_2 = new AWindow( aguix, 0, 0, 10, 10, "" );
  acsw11->add( subwin11_2, 0, 2, cmin );
  subwin11_2->create();
  std::unique_ptr< SettingsWidgets > m1 = buildW( subwin11_2 );
  acsw11->readLimits();

  AWindow *subwin12 = new AWindow( aguix, 0, 0, 10, 10, "" );
  k1->add( subwin12 );
  subwin12->create();

  std::unique_ptr< FiletypeWindow > ftw = buildFTWindow( subwin12 );

  AWindow *subwin13 = new AWindow( aguix, 0, 0, 10, 10, "" );
  k1->add( subwin13 );
  subwin13->create();

  std::unique_ptr< ArchiveWindow > arcw = buildArcWindow( subwin13 );

  str1 = catalog.getLocale( 641 );
  str1 += " - F1";
  k1->setOption( subwin11, 0, str1.c_str() );
  str1 = catalog.getLocale( 642 );
  str1 += " - F2";
  k1->setOption( subwin12, 1, str1.c_str() );
  str1 = catalog.getLocale( 643 );
  str1 += " - F3";
  k1->setOption( subwin13, 2, str1.c_str() );
  k1->maximize();
  k1->contMaximize();
  ac1->readLimits();

  AContainer *ac1_5 = ac1->add( new AContainer( win, 2, 1 ), 0, 2 );
  ac1_5->setMinSpace( 5 );
  ac1_5->setMaxSpace( -1 );
  ac1_5->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_5->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                0 ), 0, 0, cfix );
  Button *cancelb = (Button*)ac1_5->add( new Button( aguix,
                                                     0,
                                                     0,
                                                     catalog.getLocale( 8 ),
                                                     0 ), 1, 0, cfix );
  win->setDoTabCycling( true );
  win->contMaximize( true );

  w = win->getWidth();
  h = win->getHeight();

  int rx, ry, rw, rh;

  aguix->getLargestDimensionOfCurrentScreen( &rx, &ry,
                                             &rw, &rh );

  sw = rw;
  sw = (int)( (double)sw * 0.8 );
  if ( sw < 200 ) sw = 200;
  if ( w < sw ) w = sw;
  win->resize( w, h );

  win->show();
  k1->show();

  m_cmd_sg->takeFocus();

  std::string cfgfile = Worker::getWorkerConfigDir();
#ifdef USEOWNCONFIGFILES
  cfgfile = NWC::Path::join( cfgfile, "startprog-history2" );
#else
  cfgfile = NWC::Path::join( cfgfile, "startprog-history" );
#endif

  PersistentStringList history( cfgfile );
  m_current_original_file = m_cmd_sg->getText();
  m_completion_enabled = true;
  m_previous_sg_content = "";

  updateSGFromComplete( m_cmd_sg, history, m_completion_lv );

  AGMessage *msg;
  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  int ende=0;
  while(ende==0) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch ( msg->type ) {
          case AG_CLOSEWINDOW:
              if ( msg->closewindow.window == win->getWindow() ) ende = -1;
          case AG_BUTTONCLICKED:
              if ( msg->button.button == okb ) {
                  if ( m1->validate() == 1 ) ende = 1;
              } else if ( msg->button.button== cancelb ) ende = -1;
              else if ( msg->button.button == m_cmd_flag_b ) {
                  char *tstr = OwnOp::getFlag();
                  if ( tstr != NULL ) {
                      m_cmd_sg->insertAtCursor( tstr );
                      _freesafe( tstr );
                  }
              }
              break;
	case AG_STRINGGADGET_DEACTIVATE:
          if ( msg->stringgadget.sg == m1->_sg ) {
	    m1->validate();
	  }
	  break;
        case AG_FIELDLV_ONESELECT:
        case AG_FIELDLV_MULTISELECT:
          if ( msg->fieldlv.lv == arcw->_lv ) {
            row = arcw->_lv->getActiveRow();
            if ( arcw->_lv->isValidRow( row ) == true ) {
	      arcw->_sg->setText( arcw->_lv->getText( row, 0 ).c_str() );
            }
          } else if ( msg->fieldlv.lv == m_completion_lv ) {
              updateSGFromRow( msg->fieldlv.row, history );
          }
          break;
	case AG_STRINGGADGET_OK:
	  if ( msg->stringgadget.sg == m_cmd_sg ) {
	    if ( m1->validate() == 1 ) ende = 1;
	  }
	  break;
    case AG_STRINGGADGET_CURSORCHANGE:
        if ( msg->stringgadget.sg == m_cmd_sg ) {
            m_completion_enabled = false;
        }
        break;
    case AG_STRINGGADGET_CONTENTCHANGE:
        if ( msg->stringgadget.sg == ftw->m_filter_sg ) {
            ftw->updateView();
        } else if ( msg->stringgadget.sg == m_cmd_sg ) {
            updateSGFromComplete( m_cmd_sg, history, m_completion_lv );
        }
        break;
        case AG_KEYPRESSED:
          if ( win->isParent( msg->key.window, false ) == true ) {
            switch ( msg->key.key ) {
              case XK_1:
                i = m1->_cyb->getSelectedOption() + 1;
                if ( i >= 4 ) i = 0;
                m1->_cyb->setOption( i );
                break;
              case XK_2:
                m1->_gcb->setState( ( m1->_gcb->getState() == true ) ? false : true );
                break;
              case XK_3:
                m1->_ibcb->setState( ( m1->_ibcb->getState() == true ) ? false : true );
                break;
              case XK_4:
                m1->_dcdcb->setState( ( m1->_dcdcb->getState() == true ) ? false : true );
                break;
              case XK_Return:
                  if ( cancelb->getHasFocus() == false ) {
                      if ( m1->validate() == 1 ) ende = 1;
                  }
                  break;
                case XK_Escape:
                    ende = -1;
                    break;
                case XK_F1:
                    //TODO Was passiert mit Focus, wenn ich per Key die Kartei wechsle?
                    k1->optionChange( 0 );
                    break;
                case XK_F2:
                    k1->optionChange( 1 );
                    break;
                case XK_F3:
                    k1->optionChange( 2 );
                    break;
                case XK_Down:
                    if ( m_completion_lv->getElements() > 0 ) {
                        int trow = m_completion_lv->getActiveRow();

                        trow++;

                        if ( trow >= m_completion_lv->getElements() ) {
                            trow = m_completion_lv->getElements() - 1;
                        }

                        m_completion_lv->setActiveRow( trow );
                        m_completion_lv->showActive();

                        updateSGFromRow( trow, history );
                    }
                    break;
                case XK_Up:
                    if ( m_completion_lv->getElements() > 0 ) {
                        int trow = m_completion_lv->getActiveRow();

                        if ( trow > 0 ) {
                            trow--;
                        }

                        m_completion_lv->setActiveRow( trow );
                        m_completion_lv->showActive();

                        updateSGFromRow( trow, history );
                    }
                    break;
            }
          }
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }

  StartSettings *sets = new StartSettings();

  if ( ende == 1 ) {
      if ( k1->getCurOption() == 0 ) {
          sets->_startmode = StartSettings::STARTPROG;
          sets->_command = m_cmd_sg->getText();

          if ( AGUIXUtils::ends_with( sets->_command, m_current_original_file ) ) {
              std::string base = sets->_command;
              base.resize( base.length() - m_current_original_file.length() );

              AGUIXUtils::rstrip( base );

              if ( ! base.empty() ) {
                  if ( base[0] != ' ' ) {
                      history.removeEntry( base );
                      history.pushFrontEntry( base );
                  }
              }
          } else {
              std::string base = sets->_command;
              if ( AGUIXUtils::ends_with( base, "}" ) ) {
                  AGUIXUtils::rstrip( base );

                  if ( ! base.empty() ) {
                      if ( base[0] != ' ' ) {
                          history.removeEntry( base );
                          history.pushFrontEntry( base );
                      }
                  }
              }
          }

          sets->_view_str = m1->_sg->getText();
          switch ( m1->_cyb->getSelectedOption() ) {
              case 1:
                  sets->_start = STARTPROGOP_START_IN_TERMINAL;
                  break;
              case 2:
                  sets->_start = STARTPROGOP_START_IN_TERMINAL_AND_WAIT4KEY;
                  break;
              case 3:
                  sets->_start = STARTPROGOP_SHOW_OUTPUT;
                  break;
              case 4:
                  sets->_start = STARTPROGOP_SHOW_OUTPUT_INT;
                  break;
              default:
                  sets->_start = STARTPROGOP_START_NORMAL;
                  break;
          }
          sets->_global = m1->_gcb->getState();
          sets->_inbackground = m1->_ibcb->getState();
          sets->_dontcd = m1->_dcdcb->getState();
      } else if ( k1->getCurOption() == 1 ) {
          sets->_startmode = StartSettings::HANDLE_TYPE;

          row = ftw->_lv->getActiveRow();
          if ( ( ftw->_lv->isValidRow( row ) == true ) &&
               ( row < ftw->_flatlist->getNrOfEntries() ) ) {
              WCFiletype *ft = ftw->_flatlist->getEntry( row ).filetype;
              if ( ft != NULL ) {
                  sets->_type = ft;
              }
          }
      } else if ( k1->getCurOption() == 2 ) {
          sets->_startmode = StartSettings::HANDLE_ARCHIVE;
          sets->_archive = arcw->_sg->getText();
      }
  }
  delete win;

  m_cmd_sg = nullptr;
  m_cmd_flag_b = nullptr;

  return std::unique_ptr< StartSettings >( sets );
}

StartProgOp::StartSettings::StartSettings() : _startmode( CANCEL ),
                                              _type( NULL ),
                                              _start( STARTPROGOP_START_NORMAL ),
                                              _global( true ),
                                              _inbackground( false ),
                                              _dontcd( false )
{
}

std::unique_ptr< StartProgOp::FiletypeWindow > StartProgOp::buildFTWindow( AWindow *win )
{
  FiletypeWindow *m1;

  AGUIX *aguix = Worker::getAGUIX();
  FieldListView *lv;
  const int cmin = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cincwnr = cincw +
                      AContainer::ACONT_NORESIZE;
  
  AContainer *ac1 = win->setContainer( new AContainerBB( win, 1, 3 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  ac1->add( new Text( aguix, 0, 0, catalog.getLocale( 644 ) ), 0, 0, cincwnr );

  AContainer *ac1_sub1 = ac1->add( new AContainer( win, 2, 1 ),
                                   0, 1 );
  ac1_sub1->setMinSpace( 5 );
  ac1_sub1->setMaxSpace( 5 );

  ac1_sub1->add( new Text( aguix, 0, 0, catalog.getLocale( 793 ) ), 0, 0, AContainer::CO_FIX );
  auto sg = ac1_sub1->addWidget( new StringGadget( aguix, 0, 0, 50, "", 0 ),
                                 1, 0, cincw );

  lv = (FieldListView*)ac1->add( new FieldListView( aguix, 0, 0, 100, 100, 0 ), 0, 2, cmin );
  lv->setHBarState( 2 );
  lv->setVBarState( 2 );
  lv->setDisplayFocus( true );
  lv->setAcceptFocus( true );
  lv->setDefaultColorMode( FieldListView::PRECOLOR_ONLYACTIVE );

  FlatTypeList *l1 = wconfig->getFlatTypeList();
  l1->buildLV( lv, NULL );

  win->contMaximize( true );
  
  m1 = new FiletypeWindow( lv, sg, l1 );

  return std::unique_ptr< FiletypeWindow >( m1 );
}

StartProgOp::FiletypeWindow::FiletypeWindow( FieldListView *lv,
                                             StringGadget *sg,
                                             FlatTypeList *flatlist ) : _lv( lv ),
                                                                        m_filter_sg( sg ),
                                                                        _flatlist( flatlist )
{
}

StartProgOp::FiletypeWindow::~FiletypeWindow()
{
  if ( _flatlist != NULL ) delete _flatlist;
}

void StartProgOp::FiletypeWindow::updateView()
{
    delete _flatlist;

    _flatlist = new FlatTypeList( wconfig->getFiletypes(),
                                  m_filter_sg->getText() );
    _flatlist->buildLV( _lv, NULL );

    _lv->redraw();
}

std::unique_ptr< StartProgOp::ArchiveWindow > StartProgOp::buildArcWindow( AWindow *win )
{
  ArchiveWindow *m1;
  
  AGUIX *aguix = Worker::getAGUIX();
  FieldListView *lv;
  const int cmin = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cincwnr = cincw +
                      AContainer::ACONT_NORESIZE;
  int row;
  std::vector<std::string> v;
  
  AContainer *ac1 = win->setContainer( new AContainerBB( win, 1, 3 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  ac1->add( new Text( aguix, 0, 0, catalog.getLocale( 640 ) ), 0, 0, cincwnr );

  StringGadget *sg = (StringGadget*)ac1->add( new StringGadget( aguix, 0, 0, 100, "", 0 ), 0, 1, cincw );

  lv = (FieldListView*)ac1->add( new FieldListView( aguix, 0, 0, 100, 100, 0 ), 0, 2, cmin );
  lv->setHBarState( 2 );
  lv->setVBarState( 2 );
  lv->setDisplayFocus( true );
  lv->setAcceptFocus( true );

  v = AVFSSupport::getAVFSModules();

  for ( std::vector<std::string>::iterator it = v.begin(); it != v.end(); it++ ) {
    row = lv->addRow();
    lv->setText( row, 0, *it );
    lv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
  }

  win->contMaximize( true );
  m1 = new ArchiveWindow( lv, sg );

  return std::unique_ptr< ArchiveWindow >( m1 );
}

StartProgOp::ArchiveWindow::ArchiveWindow( FieldListView *lv,
					   StringGadget *sg ) : _lv( lv ),
								_sg( sg )
{
}

void StartProgOp::setGUIMsg( std::string msg )
{
  gui_msg = msg;
}

void StartProgOp::updateCompletionList( const std::string &base,
                                        PersistentStringList &history )
{

   std::list< std::string > l;

   l = history.getList();

   m_current_completions.clear();

   for ( auto &s : l ) {
       if ( AGUIXUtils::starts_with( s, base ) ) {
           m_current_completions.push_back( s );
       }
   }
}

void StartProgOp::updateSGFromComplete( StringGadget *sg,
                                        PersistentStringList &history,
                                        FieldListView *lv )
{
    std::string old_base;

    old_base = sg->getText();

    std::string prefix( old_base );

    if ( sg->getCursor() < (int)prefix.length() ) {
        prefix.resize( sg->getCursor() );
    }
    
    updateCompletionList( prefix, history );

    lv->setSize( m_current_completions.size() );

    int row = 0;
    for ( auto &s : m_current_completions ) {
        std::string s2( s );

        if ( ! AGUIXUtils::ends_with( s2, "}" ) ) {
            s2 += " ";
            s2 += m_current_original_file;
        }

        lv->setText( row, 0, s2 );
        lv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );

        row++;
    }
    lv->redraw();

    if ( ! m_completion_enabled ) return;

    if ( old_base == m_previous_sg_content ||
         old_base.length() < m_previous_sg_content.length() ) {
        m_completion_enabled = false;
        return;
    }

    if ( m_current_completions.empty() ) {
        std::string new_text( prefix );

        if ( ! AGUIXUtils::ends_with( new_text, "}" ) ) {
            new_text += " ";
            new_text += m_current_original_file;
        }

        sg->setText( new_text.c_str() );

        sg->redraw();

        m_completion_enabled = false;
        return;
    }

    if ( sg->getCursor() > 0 ) {

        std::string new_base = m_current_completions.front();

        std::string new_text( new_base );

        m_previous_sg_content = new_text;

        if ( sg->getCursor() < (int)m_previous_sg_content.length() ) {
            m_previous_sg_content.resize( sg->getCursor() );
        }

        if ( ! AGUIXUtils::ends_with( new_text, "}" ) ) {
            new_text += " ";
            new_text += m_current_original_file;
        }

        sg->setText( new_text.c_str() );

        sg->setSelection( sg->getCursor(),
                          new_text.length() );

        sg->redraw();
    }
}

void StartProgOp::updateSGFromRow( int row,
                                   PersistentStringList &history )
{
    std::string t = m_completion_lv->getText( row, 0 );

    m_cmd_sg->setText( t.c_str() );
    m_cmd_sg->setSelection( m_cmd_sg->getCursor(),
                            t.length() );
    m_cmd_sg->redraw();

    m_completion_enabled = false;

    updateSGFromComplete( m_cmd_sg, history, m_completion_lv );
}

