// ---------------------------------------------------------------------------
// - Rpi.cpp                                                                 -
// - afnix:mth module - real permutation interface implementation            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2015 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Rpi.hpp"
#include "Mthsid.hxx"
#include "Vector.hpp"
#include "Integer.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
#include "OutputStream.hpp"
 
namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------
  

  // this procedure returns a new permutation for deserialization
  static Serial* mksob (void) {
    return new Rpi;
  }
  // register the anonymous permutation serial id
  static const t_byte SERIAL_ID = Serial::setsid (SERIAL_ARPI_ID, mksob);

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default permutation

  Rpi::Rpi (void) {
    d_size = 0LL;
    p_prow = nilp;
    p_revp = nilp;
  }

  // create a permutation by size

  Rpi::Rpi (const t_long size) {
    // check the size
    if (size < 0) {
      throw Exception ("size-error", "invalid permutation size");
    }
    d_size = size;
    p_prow = new t_long[d_size];
    p_revp = new t_long[d_size];
    clear ();
  }

  // copy construct this object

  Rpi::Rpi (const Rpi& that) {
    that.rdlock ();
    try {
      d_size = that.d_size;
      p_prow = (p_prow == nilp) ? nilp : new t_long[d_size];
      p_revp = (p_revp == nilp) ? nilp : new t_long[d_size];
      for (t_long k = 0LL; k < d_size; k++) {
	p_prow[k] = that.p_prow[k];
	p_revp[k] = that.p_revp[k];
      }
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // destroy this object

  Rpi::~Rpi (void) {
    delete [] p_prow;
    delete [] p_revp;
  }

  // assign an object to this one

  Rpi& Rpi::operator = (const Rpi& that) {
    // check for self-assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      d_size = that.d_size;
      p_prow = (p_prow == nilp) ? nilp : new t_long[d_size];
      p_revp = (p_revp == nilp) ? nilp : new t_long[d_size];
      for (t_long k = 0LL; k < d_size; k++) {
	p_prow[k] = that.p_prow[k];
	p_revp[k] = that.p_revp[k];
      }
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }

  // return the class name

  String Rpi::repr (void) const {
    return "Rpi";
  }

  // return the anonymous rpi serial code

  t_byte Rpi::serialid (void) const {
    return SERIAL_ARPI_ID;
  }

  // serialize this object
  
  void Rpi::wrstream (OutputStream& os) const {
    rdlock ();
    try {
      // write the array size
      mth_wrlong (d_size, os);
      // write the array data
      for (long k = 0; k < d_size; k++) mth_wrlong (p_prow[k], os);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // deserialize this object

  void Rpi::rdstream (InputStream& is) {
    wrlock ();
    try {
      // reset the permutation
      reset ();
      // get the array size
      d_size = mth_rdlong (is);
      p_prow = (d_size == 0LL) ? nilp : new t_long[d_size];
      p_revp = (d_size == 0LL) ? nilp : new t_long[d_size];
      // get the array data
      for (long k = 0LL; k < d_size; k++) {
	p_prow[k] = mth_rdlong (is);
	p_revp[p_prow[k]] = k;
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // reset this permutation

  void Rpi::reset (void) {
    wrlock ();
    try {
      d_size = 0LL;
      delete [] p_prow; p_prow = nilp;
      delete [] p_revp; p_revp = nilp;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // clear this permutation

  void Rpi::clear (void) {
    wrlock ();
    try {
      for (t_long k = 0; k < d_size; k++) p_revp[k] = p_prow[k] = k;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the permutation size

  t_long Rpi::getsize (void) const {
    rdlock ();
    try {
      t_long result = d_size;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the permutation by position

  void Rpi::set (const t_long pos, const t_long val) {
    wrlock ();
    try {
      // check valid bounds
      if ((pos < 0LL) || (pos >= d_size) || (val < 0LL) || (val >= d_size)) {
	throw Exception ("rpi-error", "invalid permutation arguments");
      }
      // set the permutation
      p_prow[pos] = val;
      p_revp[val] = pos;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the permutation by position

  t_long Rpi::get (const t_long pos) const {
    wrlock ();
    try {
      // check valid bounds
      if ((pos < 0LL) || (pos >= d_size)) {
	throw Exception ("rpi-error", "invalid permutation arguments");
      }
      t_long result = p_prow[pos];
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the reverse permutation by position

  t_long Rpi::rget (const t_long pos) const {
    wrlock ();
    try {
      // check valid bounds
      if ((pos < 0LL) || (pos >= d_size)) {
	throw Exception ("rpi-error", "invalid permutation arguments");
      }
      t_long result = p_revp[pos];
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }


  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 5;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_SET     = zone.intern ("set");
  static const long QUARK_GET     = zone.intern ("get");
  static const long QUARK_CLEAR   = zone.intern ("clear");
  static const long QUARK_RESET   = zone.intern ("reset");
  static const long QUARK_REVERSE = zone.intern ("reverse");

  // create a new object in a generic way

  Object* Rpi::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // check for 0 argument
    if (argc == 0) return new Rpi;
    // check for 1 argument
    if (argc == 1) {
      t_long size = argv->getlong (0);
      return new Rpi (size);
    }
    // invalid arguments
    throw Exception ("argument-error", 
		     "invalid arguments with real permutation object");
  }

  // return true if the given quark is defined

  bool Rpi::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true){
      unlock ();
      return true;
    }
    bool result = hflg ? Serial::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }
  
  // apply this object with a set of arguments and a quark
  
  Object* Rpi::apply (Runnable* robj, Nameset* nset, const long quark,
		      Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_CLEAR) {
	clear ();
	return nilp;
      }
      if (quark == QUARK_RESET) {
	reset ();
	return nilp;
      }
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_GET) {
	t_long pos = argv->getlong (0);
	return new Integer (get (pos));
      }
      if (quark == QUARK_REVERSE) {
	t_long pos = argv->getlong (0);
	return new Integer (rget (pos));
      }
    }
    // dispatch 2 arguments
    if (argc == 2) {
      if (quark == QUARK_SET) {
	t_long pos = argv->getlong (0);
	t_long val = argv->getlong (1);
	set (pos, val);
	return nilp;
      }
    }
    // call the serial object
    return Serial::apply (robj, nset, quark, argv);
  }
}

