/*
 * aurora - Communications with Magnetek Aurora Inverter
 *
 * Copyright (C) 2006-2011 Curtis J. Blank curt@curtronics.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program may be used and hosted free of charge by anyone for
 * personal purposes as long as this and all included copyright
 * notice(s) remain intact.
 *
 * Obtain permission before selling the code for this program or
 * hosting this software on a commercial website, excluding if it is 
 * solely hosted for distribution, that is allowable . In all cases
 * copyright must remain intact.
 *
 * This work based on Magnetek's 'Aurora PV Inverter - Communications
 * Protocol -' document, Rel. 1.8 09/05/2005
 * Starting with v1.5-0 this work based on Power One's 'Aurora Inverter
 * Series - Communication Protocol -' document, Rel. 4.6 25/02/09
 *
 * Special thanks to Tron Melzl at Magnetek for all his help including,
 * but not limited to, supplying the Communications Protocol document
 *
 * modified 17-jul-2006 cjb	1. v1.2-6 Last 7 Days production value has been dropped 
 *				   in the v2.3 Communications Protocol doc
 * modified 27-jul-2006 cjb	1. add bVerbose in ClrSerLock
 * modified 13-oct-2006 cjb	1. make this v1.3-0
 * modified 25-apr-2007 cjb	1. take into account Daylight Savings Time when setting the Aurora's time
 * modified 29-dec-2008 cjb	1. correct an error in strftime that only may show up in the first or last
 * 				   week of the year (%G vs %Y)
 * modified 19-aug-2009 cjb	1. fix determining if serial port is in use
 * modified 22-aug-2009 cjb	1. disable XON/XOFF flow control on output
 * modified 29-sep-2009 cjb	1. don't set lckCNT = -1 when clearing stale serial port lock
 * modified 12-oct-2009 cjb	1. add -o option to output data to a file
 * modified 25-oct-2009 cjb	1. serial port configuration tweaks
 * modified 17-oct-2010 cjb	1. added -x option to enable XON/XOFF
 * modified 07-mar-2010 cjb	1. added -Y retries option and -y option to report the number of attempt made
 * modified 13-mar-2010 cjb	1. added -P option to delay read after sending command to inverter
 *				2. rename -P to -U and add -u to report it
 *				3. changed -U seconds to wait to milli-seconds to wait
 * modified 28-mar-2010 cjb	1. working on adding more DSP information
 * modified 27-jul-2010 cjb	1. added -P option to throttle commands sent to the inverter
 * modified 21-sep-2010 cjb	1. added -A option to read "Last four alarms"
 * modified 22-jan-2011 cjb	1. added -k option to get up to a years worth of daily generation history
 * modified 19-mar-2011 cjb	1. added -X option to enable RTS/CTS on the serial port
 * modified 13-aug-2001 cjb	1. increase the allowed values from 31 to 63 for inverter address (-a)
 *                                 as allowed in newer inverters
 * modified 22-aug-2011 cjb	1. clean up compile 'unused-but-set' messages
 * modified 05-nov-2011 cjb	1. remove size restriction on monetary identifier of the -C option
 *                              2. added -q --energy-sent option
 * modified 09-nov-2011 cjb	1. added -L --adjust-time option
 *
 */

static char	VersionM[] = "1.7.3";
char     VersionC[7];

#include <syscall.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <termios.h>
#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <getopt.h>
#include <time.h>
#include <errno.h>
#include <error.h>
#include "include/main.h"
#include "include/comm.h"
#include "include/names.h"

BOOL bVerbose = FALSE;
BOOL bColumns = FALSE;		/* Output data in columns */
int yGetDSP = -1;		/* Measure request to the DSP */
BOOL bGetDSPExtended = FALSE;	/* Measure request to the DSP more parameters */
BOOL bGetDSP3Phase = FALSE;	/* Measure request to the DSP three-phase parameter */
BOOL bHideDSP = FALSE;
BOOL bGetEnergy = FALSE;	/* Cumulated Energy Readings */
BOOL bGetInvTime = FALSE;	/* Display Inverter time flag */
BOOL bGetLocTime = FALSE;	/* Display computer time flag */
BOOL bCommCheck = FALSE;
BOOL bColOutput = FALSE;
BOOL bCalcGridPwr = FALSE;
BOOL bXonXoff = FALSE;
BOOL bRTSCTS = FALSE;
BOOL bRptReties = FALSE;
BOOL bRptReadPause = FALSE;
int yTimeout = 0;		/* read timeout value in us */
long unsigned int PID;
int yMaxAttempts = 1;
int yReadPause = 0;
long int yCommPause = 0;
int yCheckSetTime = -1;		/* Check the time on the inverter and if different then system time set it */
struct timeval lastcommtv;

FILE *outfp;

char RunTime[18] = " ";
char VersionSHc[6];
int ModelID = 0;

float yCost = 0.0;
char *sCostType = NULL;

/* local Data */
static char VersionCHc[6] = VersionCH;
static char VersionMHc[6] = VersionMH;
static char VersionNHc[6] = VersionNH;
static int yAddress = 0;			/* Inverter address 1-63 */
static char *szttyDevice = NULL;
static char *outfile = NULL;
static BOOL bHelp = FALSE;			/* Print the help text */
static BOOL bGetPN = FALSE;			/* Get Inverter Part Number */
static BOOL bGetVer = FALSE;			/* Get version string */
static BOOL bGetVerFW = FALSE;			/* Get firmware version string */
static BOOL bGetSN = FALSE;			/* Get Serial Number */
static BOOL bGetMfg = FALSE;			/* Get Manufacturing Date */
static BOOL bGetConf = FALSE;			/* Get System Configuration */
static BOOL bGetState = FALSE;			/* State request */
static BOOL bGetJoules = FALSE;			/* Energy cumulated in the last 10 seconds */
static BOOL bGetCount = FALSE;			/* Get Time Counters */
static BOOL bSetTime = FALSE;			/* Set time flag */
static BOOL bVersion = FALSE;			/* Display program version */
static BOOL bGetTime;
static BOOL bGetLastAlarms = FALSE;		/* Get last four alarms */
static BOOL bFileError = FALSE;
static int yGetEnergyDaily = 0;
static int yGetEnergySent = 0;
static unsigned char yDelay;			/* Read wait time */
static int yLockWait = 0;			/* Seconds to wait to lock serial port */
char *devLCKfile = NULL;
char *devLCKfileNew = NULL;
static struct termios oldtio;			/* current serial device configuration */

/* local functions */
static int GetParms(int argc, char *argv[]);
static void *getMemPtr(size_t mSize);
static void Version();

/*--------------------------------------------------------------------------
    main
----------------------------------------------------------------------------*/
int main(int argc, char *argv[])
{
    int fdser;				/* serial device fd */
    FILE *fdserlck = NULL;
    struct termios newtio;		/* serial device configuration */
    int rc = 0;
    time_t timeValue;
    struct tm timStruct;
    long unsigned int rPID;
    char sPID[10];
    int bRead, bWrite, lckCNT;
    int errno_save = 0;
    int fLen = 0;
    char *cmdFile = NULL;
    char *command = NULL;
    char *SubStrPos = NULL;

    lastcommtv.tv_sec = 0;
    lastcommtv.tv_usec = 0;
    timeValue = time(NULL);
    timStruct = *(localtime(&timeValue));
    strftime(RunTime,18,"%Y%m%d-%H:%M:%S ",&timStruct);

    PID = getpid();

    /* Get command line parms */
    if ((!GetParms(argc, argv) && !bVersion) || bHelp) {
        printf("\nSee http://www.curtronics.com/Solar/ for Solar Site example\n\n");
        printf("Usage: %s [Options] Device\t\t\tv%s\n\n",ProgramName,VersionM);
        printf("Options:\n");
        printf(" -A, --last-alarms              Get last four alarms (once displayed FIFO queue is cleared)\n");
        printf(" -b, --verbose                  Verbose mode. For maximum effectiveness should be the first option\n");
        printf("                                  on the command line\n");
        printf(" -C, --calc-value=<num[:$]>     Calculate monetary value using <num> * kWh. \":$\" is optional and if\n");
        printf("                                  included the character(s) represented by the \"$\" will be used to\n");
        printf("                                  denote monetary type in the output. Defaults to \"$\"\n");
        printf(" -c, --columnize                Output data in columns --> for -d, -e, -D, -E, -t, & -T options\n");
        printf("                                  only will disable any other options -- if value ends with an\n");
        printf("                                  \"*\" reporting of that item may not be in inverters firmware\n");
        printf(" -d <num>, --get-dsp=<num>      Get DSP data. <num> indicates string to get data for. 0 indicates\n");
        printf("                                  both, 1 for only string 1, 2 for only string 2. <num> is required\n");
        printf("                                  for short option and <num> is optional for long option and if\n");
        printf("                                  omitted for long option then data for both input strings will be\n");
        printf("                                  retrieved.\n");
        printf(" -D, --get-dsp-extended         Get more DSP data\n");
        printf(" -e, --get-energy               Get Cumulated Energy readings\n");
        printf(" -E, --get-dsp-3phase           Get 3-Phase DSP data\n");
        printf(" -f, --firmware-ver             Query for Firmware Version string\n");
        printf(" -g, --mfg-date                 Query for Inverter Manufacturing Date\n");
        printf(" -h, --help                     This text\n");
        printf(" -i, --get-count                Display Inverter Time Counters\n");
        printf(" -j, --get-joules               Display Energy cumulated in the last 10 seconds\n");
        printf(" -k <num>, --daily-kwh=<num>    Get past daily kWh for <num> days (1-366) ** Experimental **\n");
        printf("                                  <num> is required for short option and optional for long option\n");
        printf("                                  and if omitted for long option then all 366 days or as many that\n");
        printf("                                  are found will be displayed\n");
        printf(" -L <num>, --adjust-time=<num>  Automatically adjust the inverter time if it differs from the computer\n");
        printf("                                  time. If <num> is 0 (zero) do a Daylight Savings Time check. If <num>\n");
        printf("                                  is >= 1 change the inverter time if it differs by <num> or more seconds.\n");
        printf("                                  See the README for more information and constraints\n");
        printf(" -l <num>, --delay=<num>        Serial port timeout in for read. <num> in 1/10ths seconds\n");
        printf("                                  Default is 1 (0.1 sec)\n");
        printf(" -m, --get-conf                 Query for Inverter System Configuration\n");
        printf(" -n, --serial-number            Query for Inverter Serial Number\n");
        printf(" -o, --output-file=<filename>   Append data to file (Created if nonexistant)\n");
        printf(" -P <num>, --comm-pause=<num>   Wait <num> us between sending commands to inverter (1-1000000)\n");
        printf(" -p, --part-number              Query for Inverter Part Number\n");
        printf(" -q <num>, --energy-sent=<num>  Get past energy delivered to the grid ** Experimental **\n");
        printf("                                  in 10 second intervals for <num> minutes (1-1440). <num> is\n");
        printf("                                  optional for long option and if omitted all data, ~24 hours\n");
        printf("                                  worth will be reported. It is suggested the -Y option be used\n");
        printf("                                  with this due to the extensive length of time this could take.\n");
        printf("                                  See the README file for important information on this option\n");
	printf(" -R <num>, --read-timeout=<num> Timeout value when reading data from the Inverter (ms)\n");
        printf(" -r, --calc-grid-power          Calc Grid power using Grid Voltage * Grid Current,\n");
        printf("                                  instead of reporting the Inverter's value. --> for \n");
        printf("                                  -d option only, ignored when used with -c option\n");
        printf("                                  (Inverter typically reports a lower value. This\n");
        printf("                                  affects Inverter conversion efficiency value.)\n");
        printf(" -S, --set-time                 Set Inverter Date/Time to system time\n");
        printf(" -s, --get-state                Get Inverter State\n");
        printf(" -T, --get-loctime              Display computer Date/Time\n");
        printf(" -t, --get-invtime              Display Inverter Date/Time\n");
        printf(" -V, --version                  Aurora communications program version\n");
        printf(" -v, --inv-version              Query for Version string\n");
	printf(" -U <num>, --read-pause=<num>   Pause <num> milli-seconds after sending command to inverter before\n");
        printf("                                  reading response from inverter (1-10000)\n");
        printf(" -u, --rpt-read-pause           Report when/that pausing before read\n");
        printf(" -w, --lock-wait                Seconds to wait to lock serial port. (1-30)\n");
        printf(" -X, --rts-cts                  Enable RTS/CTS on the serial port.\n");
	printf(" -x, --xon-xoff                 Enable XON/XOFF on the serial port.\n");
        printf(" -Y <num>, --retries=<num>      Retry failed communications with inverter up to <num> times (1-100)\n");
        printf(" -y, --rpt-retries              Report the number of retires done\n\n");
        printf("               *** Required Parameters ***\n");
        printf(" -a <num>, --address=<num>      Inverter address. 1-31 on older inverters, 1-63 on newer inverters.\n");
        printf(" Device                         Serial Device.\n");
        printf("\n");
        if (bHelp) 
            exit(0);
        else
            exit(2);
    }

    if (bVerbose) fprintf(stderr, "\nRunTime %s v%-6s\n",RunTime,VersionM);

    if (yCost > 0.0 && sCostType == NULL) {
        sCostType = getMemPtr(2);
        strcpy(sCostType,"$");
        if (bVerbose) fprintf (stderr, "Monetary type \"%s\"\n",sCostType);
    }

    if (bVerbose) fprintf (stderr, "PID : %lu\n", PID);

    if (bVersion) {
        Version();
        exit(0);
    }

    if (bVerbose) fprintf(stderr, "\nAttempting to get lock on Serial Port %s...\n",szttyDevice);
    fdserlck = fopen(devLCKfile, "a");
    if (fdserlck == NULL) {
        if (bVerbose) fprintf(stderr, "\n");
        fprintf(stderr, "%s: %s: Problem locking serial device, can't open lock file: %s for write.\n\n",RunTime,ProgramName,devLCKfile);
        exit(2);
    }
    bWrite = fprintf(fdserlck, "%lu\n", PID);
    errno_save = errno;
    fclose(fdserlck);
    fdserlck = NULL;
    if (bWrite < 0 || errno_save != 0) {
        if (bVerbose) fprintf(stderr, "\n");
        fprintf(stderr, "%s: %s: Problem locking serial device, can't write lock file: %s.\n%s\n\n",RunTime,ProgramName,devLCKfile,strerror (errno_save));
        exit(2);
    }

    rPID = 0;
    lckCNT = -1;
    while(rPID != PID && lckCNT++ < yLockWait) {
        SubStrPos = NULL;
        fdserlck = fopen(devLCKfile, "r");
        if (fdserlck == NULL) {
            if (bVerbose) fprintf(stderr, "\n");
            fprintf(stderr, "%s: %s: Problem locking serial device, can't open lock file: %s for read.\n\n",RunTime,ProgramName,devLCKfile);
            exit(2);
        }
        bRead = fscanf(fdserlck, "%lu", &rPID);
        errno_save = errno;
        fclose(fdserlck);
        fdserlck = NULL;
        if (bRead == EOF || errno_save != 0) {
            if (bVerbose) fprintf(stderr, "\n");
            fprintf(stderr, "%s: %s: Problem locking serial device, can't read lock file: %s.\n%s\n\n",RunTime,ProgramName,devLCKfile,strerror (errno_save));
            exit(2);
        }

        sPID[0] = '\0';
        sprintf(sPID,"%lu",rPID);
        cmdFile = getMemPtr(strlen(sPID)+14+1);
        cmdFile[0] = '\0';
        sprintf(cmdFile,"/proc/%lu/cmdline",rPID);
        if (bVerbose) fprintf(stderr, "\nChecking process %lu for lock\n",rPID);
        fdserlck = fopen(cmdFile, "r");
        if (fdserlck != NULL) {
            fLen = 0;
            while (fgetc(fdserlck) != EOF) fLen++;
            if (fLen > 0) {
                command = getMemPtr(fLen+1);
                command[0] = '\0';
                rewind(fdserlck);
                bRead = fscanf(fdserlck, "%s", command);
            }
            fclose(fdserlck);
            fdserlck = NULL;
            if (command != NULL) SubStrPos = strstr(command, ProgramName);
        }
        if (cmdFile != NULL) {
            free(cmdFile);
            cmdFile = NULL;
        }
        if (bVerbose) {
            fprintf (stderr, "rPID: %lu SubStrPos: %s command: %s",rPID,SubStrPos,command);
            if (rPID == PID) fprintf (stderr, " = me");
            fprintf (stderr, "\n");
        }
        if (command != NULL) {
            free(command);
            command = NULL;
        }
        if (rPID != PID) {
             if (SubStrPos == NULL) {
                 if (bVerbose) fprintf (stderr, "\n");
                 fprintf(stderr, "%s: %s: Clearing stale serial port lock. (%lu)\n",RunTime,ProgramName,rPID);
                 ClrSerLock(rPID);
             } else if (yLockWait > 0)
                 sleep(1);
        }
    }
    if (bVerbose && rPID == PID) fprintf(stderr, "Appears we got the lock.\n");
    if (rPID != PID) {
        ClrSerLock(PID);
        if (bVerbose) fprintf (stderr, "\n");
        fprintf(stderr, "%s: %s: Problem locking serial device %s, couldn't get the lock for %lu, locked by %lu.\n\n",RunTime,ProgramName,szttyDevice,PID,rPID);
        exit(2);
    }

    if (bVerbose) fprintf(stderr, "\nOpening Serial Port %s...  ", szttyDevice);
    fdser = open(szttyDevice, O_RDWR | O_NOCTTY );
    if (fdser < 0) {
        ClrSerLock(PID);
        if (bVerbose) fprintf(stderr, "\n");
        fprintf(stderr, "%s: %s: Problem opening serial device, check device name.\n\n",RunTime,ProgramName);
        exit(2);
    }
    if (bVerbose) fprintf(stderr, "Serial Port %s successfully opened.\n", szttyDevice);

    tcgetattr(fdser, &oldtio);      /* save previous port settings */

    memset(&newtio, 0, sizeof(newtio));

    newtio.c_cflag &= ~PARENB;			/* no parity */
    newtio.c_cflag &= ~CSTOPB;			/* on stop bit */
    newtio.c_cflag &= ~CSIZE;			/* character size mask */
    newtio.c_cflag &= ~HUPCL;			/* no hangup */
    if (bRTSCTS)
        newtio.c_cflag |= ~CRTSCTS;             /* enable hardware flow control */
    else
        newtio.c_cflag &= ~CRTSCTS;             /* disable hardware flow control */
    newtio.c_cflag |= CS8 | CLOCAL | CREAD;	/* 8 bit - ignore modem control lines - enable receiver */
    if (bXonXoff)
        newtio.c_iflag |= IXON | IXOFF;		/* enable XON/XOFF flow control on output & input */
    else {
        newtio.c_iflag &= ~IXON;		/* disable XON/XOFF flow control on output */
        newtio.c_iflag &= ~IXOFF;		/* disable XON/XOFF flow control on input */
    }
    newtio.c_iflag |= IGNBRK | IGNPAR;		/* ignore BREAK condition on input & framing errors & parity errors */
    newtio.c_oflag = 0;	    			/* set serial device input mode (non-canonical, no echo,...) */
    newtio.c_oflag &= ~OPOST;			/* enable output processing */
    newtio.c_lflag = 0;
    newtio.c_cc[VTIME]    = yDelay;		/* timeout in 1/10 sec intervals */
    newtio.c_cc[VMIN]     = 0;			/* block until char or timeout */

    if (cfsetospeed (&newtio, B19200)) {
        if (bVerbose) fprintf(stderr, "\n");
        fprintf(stderr, "\n%s: %s: Problem setting serial output speed.\n\n",RunTime,ProgramName);
        if (bVerbose) fprintf(stderr, "Closing Serial Port %s...",szttyDevice);
        if (close(fdser)) {
            if (bVerbose) fprintf(stderr, "\n");
            fprintf(stderr, "%s: %s: Problem closing serial device.\n",RunTime,ProgramName);
        }
        if (bVerbose) { fprintf(stderr, " Success!\n"); }
        ClrSerLock(PID);
        exit(2);
    }
    if (cfsetispeed (&newtio, B19200)) {
        if (bVerbose) fprintf(stderr, "\n");
        fprintf(stderr, "\n%s: %s: Problem setting serial input speed.\n\n",RunTime,ProgramName);
        if (bVerbose) fprintf(stderr, "Closing Serial Port %s...",szttyDevice);
        if (close(fdser)) {
            if (bVerbose) fprintf(stderr, "\n");
            fprintf(stderr, "%s: %s: Problem closing serial device.\n",RunTime,ProgramName);
        }
        if (bVerbose) { fprintf(stderr, " Success!\n"); }
        ClrSerLock(PID);
        exit(2);
    }

    if (bVerbose) { fprintf(stderr, "Configuring serial device... Flushing unread data first... "); }
    errno = 0;
    if (tcflush(fdser, TCIFLUSH)) {
        if (bVerbose) fprintf(stderr, "\n");
        fprintf(stderr, "\n%s: %s: Problem flushing serial device: (%i) %s\n\n",RunTime,ProgramName,errno,strerror(errno));
    }
    if (tcsetattr(fdser, TCSANOW, &newtio)) {
        if (bVerbose) fprintf(stderr, "\n");
        fprintf(stderr, "\n%s: %s: Problem configuring serial device.\n\n",RunTime,ProgramName);
        if (bVerbose) fprintf(stderr, "Closing Serial Port %s...",szttyDevice);
        if (close(fdser)) {
            if (bVerbose) fprintf(stderr, "\n");
            fprintf(stderr, "%s: %s: Problem closing serial device.\n",RunTime,ProgramName);
        }
        if (bVerbose) { fprintf(stderr, " Success!\n"); }
        ClrSerLock(PID);
        exit(2);
    }

    if (bVerbose) { fprintf(stderr, " Success!\nFlushing serial device buffer..."); }

    errno = 0;
    if (tcflush(fdser, TCIOFLUSH)) {
        if (bVerbose) fprintf(stderr, "\n");
        fprintf(stderr, "\n%s: %s: Problem flushing serial device: (%i) %s\n\n",RunTime,ProgramName,errno,strerror(errno));
        if (bVerbose) fprintf(stderr, "Closing Serial Port %s...",szttyDevice);
        if (close(fdser)) {
            if (bVerbose) fprintf(stderr, "\n");
            fprintf(stderr, "%s: %s: Problem closing serial device.\n",RunTime,ProgramName);
        }
        if (bVerbose) { fprintf(stderr, " Success!\n"); }
        ClrSerLock(PID);
        exit(2);
    }

    if (bVerbose) { fprintf(stderr, " Success!\n"); }

    bCommCheck = TRUE;
    if (bVerbose) { fprintf( stderr, "\nComm Check: Let's see if the Aurora is listening... "); }
    outfp = stderr;
    if (CommCheck(fdser,yAddress) >= 0) {
        if (bVerbose) fprintf(stderr, "Comm Check: OK\n");
    } else {
        if (bVerbose)
             fprintf(stderr, "Comm Check: Failure, aborting...\n");
        else {
             if (bRptReadPause) fprintf(stderr, "\n");
             fprintf(stderr, "%s: %s: No response after %i attempts\n",RunTime,ProgramName,yMaxAttempts);
        }
        rc = -1;
    }
    bCommCheck = FALSE;

    if (rc == 0 && bSetTime) {
        if (!bGetInvTime) bGetLocTime = TRUE;
        rc = SetTime(fdser,yAddress,FALSE);
    }

    if (rc == 0 && yCheckSetTime >= 0) {
        rc = CheckSetTime(fdser,yAddress);
    }

    if (outfile != NULL) {
        if (! (outfp = fopen(outfile, "a"))) {
            fprintf(stderr, "%s: %s: Problem opening output file %s\n",RunTime,ProgramName,outfile);
            bFileError = TRUE;
        }
    } else {
        outfp = stdout;
    }

    if (! bFileError) {

        bGetTime = bGetInvTime | bGetLocTime;

        if (rc == 0 && bGetTime)
            rc |= GetTime(fdser,yAddress);

        if (!bColumns) {

            if (rc == 0 && bGetPN)
                rc |= GetPN(fdser,yAddress);

            if (rc == 0 && bGetSN)
                rc |= GetSN(fdser,yAddress);

            if (rc == 0 && bGetVerFW)
                rc |= GetVerFW(fdser,yAddress);

            if (rc == 0 && bGetMfg)
                rc |= GetMfgDate(fdser,yAddress);

            if (rc == 0 && bGetVer)
                rc |= GetVer(fdser,yAddress);

            if (rc == 0 && bGetConf)
                rc |= GetConf(fdser,yAddress);

            if (rc == 0 && bGetJoules)
                rc |= GetJoules(fdser,yAddress);

            if (rc == 0 && bGetState)
                rc |= GetState(fdser,yAddress);

            if (rc == 0 && bGetCount)
                rc |= GetCounters(fdser,yAddress);

            if (rc == 0 && bGetLastAlarms)
                rc |= GetLastAlarms(fdser,yAddress);

            if (rc == 0 && yGetEnergyDaily > 0)
                rc |= GetCEDaily(fdser,yAddress,yGetEnergyDaily);

            if (rc == 0 && yGetEnergySent > 0)
                rc |= GetCESent(fdser,yAddress,yGetEnergySent);

        }

        if (rc == 0 && yGetDSP >= 0)
            rc |= GetDSP(fdser,yAddress);

        if (rc == 0 && bGetEnergy)
            rc |= GetCE(fdser,yAddress);

        if (rc == 0 && bGetDSPExtended)
            rc |= GetDSPExtended(fdser,yAddress);

        if (rc == 0 && bGetDSP3Phase)
            rc |= GetDSP3Phase(fdser,yAddress);

    }

    if (rc >= 0) {
        if (bColumns && bColOutput)
            fprintf(outfp, "  OK\n");
        else
            fprintf(outfp, "\n");
        if (bVerbose) fprintf(stderr, "Complete.\n\n");
    } else if (bColumns && bColOutput)
        fprintf(outfp, "\n");

    if (bVerbose) fprintf(stderr, "rc: %d\n\n",rc);

    if (outfile != NULL && ! bFileError) fclose(outfp);


    /* all done, exit */

    RestorePort(fdser);
    ClrSerLock(PID);

    if (bVerbose) fprintf(stderr, "\nComplete.\n\n");

    if (rc >= 0) exit(0);
    exit(1);
}

/*--------------------------------------------------------------------------
    RestorePort
    Restore Serial Port Settings
----------------------------------------------------------------------------*/
int RestorePort(int fdser) {

    if (bVerbose) fprintf(stderr, "Restoring Serial Port settings %s...", szttyDevice);
    if (tcsetattr(fdser, TCSANOW, &oldtio)) {		/* restore previous port settings */
        if (bVerbose) fprintf(stderr, "\n");
        fprintf(stderr, "%s: %s: Problem restoring serial device settings.\n",RunTime,ProgramName);
        if (bVerbose) fprintf(stderr, "Closing Serial Port %s...",szttyDevice);
        if (close(fdser)) {
            if (bVerbose) fprintf(stderr, "\n");
            fprintf(stderr, "%s: %s: Problem closing serial device, check device name.\n",RunTime,ProgramName);
        }
        if (bVerbose) fprintf(stderr, " Success!\n");
        return 2;
    }

    if (bVerbose) { fprintf(stderr, " Success!\nFlushing serial device buffer..."); }

    errno = 0;
    if (tcflush(fdser, TCIOFLUSH)) {
        if (bVerbose) fprintf(stderr, "\n");
        fprintf(stderr, "\n%s: %s: Problem flushing serial device: (%i) %s\n\n",RunTime,ProgramName,errno,strerror(errno));
        if (bVerbose) fprintf(stderr, "Closing Serial Port %s...",szttyDevice);
        if (close(fdser)) {
            if (bVerbose) fprintf(stderr, "\n");
            fprintf(stderr, "%s: %s: Problem closing serial device.\n",RunTime,ProgramName);
        }
        if (bVerbose) { fprintf(stderr, " Success!\n"); }
        return 2;
    }

    if (bVerbose) fprintf(stderr, " Success!\nClosing Serial Port %s...", szttyDevice);

    if (close(fdser)) {
        if (bVerbose) fprintf(stderr, "\n");
        fprintf(stderr, "%s: %s: Problem closing serial device.\n",RunTime,ProgramName);
        return 2;
    }
    if (bVerbose) fprintf(stderr, " Success!\n");
    return 0;
}


/*--------------------------------------------------------------------------
    ClrSerLock
    Clear Serial Port lock.
----------------------------------------------------------------------------*/
int ClrSerLock(long unsigned int PID) {
    FILE *fdserlck, *fdserlcknew;
    long unsigned int rPID;
    int bWrite, bRead;
    int errno_save = 0;


    if (bVerbose) fprintf(stderr, "\ndevLCKfile: <%s>\ndevLCKfileNew: <%s>\nClearing Serial Port Lock (%lu)...", devLCKfile, devLCKfileNew, PID);
    fdserlck = fopen(devLCKfile, "r");
    if (fdserlck == NULL) {
        if (bVerbose) fprintf(stderr, "\n");
        fprintf(stderr, "\n%s: %s: Problem opening serial device lock file to clear PID %lu: %s for read.\n\n",RunTime,ProgramName,PID,devLCKfile);
        return(0);
    }
    fdserlcknew = fopen(devLCKfileNew, "w");
    if (fdserlcknew == NULL) {
        if (bVerbose) fprintf(stderr, "\n");
        fprintf(stderr, "\n%s: %s: Problem opening new serial device lock file to clear PID %lu: %s for write.\n\n",RunTime,ProgramName,PID,devLCKfileNew);
        fclose(fdserlck);
        return(0);
    }
    bRead = fscanf(fdserlck, "%lu", &rPID);
    while (bRead != EOF) {
        if (rPID != PID) {
            errno = 0;
            bWrite = fprintf(fdserlcknew, "%lu\n", rPID);
            errno_save = errno;
            if (bWrite < 0 || errno_save != 0) {
                fprintf(stderr, "\n%s: %s: Problem clearing serial device lock, can't write lock file: %s.\n%s\n\n",RunTime,ProgramName,devLCKfile,strerror (errno_save));
                fclose(fdserlcknew);
                return(0);
            }
        }
        bRead = fscanf(fdserlck, "%lu", &rPID);
    }
    fclose(fdserlck);
    fclose(fdserlcknew);
    errno = 0;
    if (rename(devLCKfileNew,devLCKfile)) fprintf(stderr, "\n%s: %s: Problem clearing serial device lock, can't update lock file: %s.\n%s\n\n",RunTime,ProgramName,devLCKfile,strerror (errno));
    if (bVerbose) fprintf(stderr, " done.\n");

    return -1;
}


/*--------------------------------------------------------------------------
    isNumeric
----------------------------------------------------------------------------*/
BOOL isNumeric(char *p)
{
    int i;

    for (i = 0; i <= strlen(p); i++)
        if (p[i] != '\0' && ! isdigit(p[i])) return(FALSE);
    return(TRUE);
}

/*--------------------------------------------------------------------------
    GetParms
    Reads command line parameters.
----------------------------------------------------------------------------*/
int GetParms(int argc, char *argv[])
{
    extern char *optarg;
    extern int optind, opterr, optopt;
    int c;
    int i = 0;
    BOOL b = FALSE;
    char *pos;
    char *SubStrPos = NULL;
    char sPID[10];
    static char *Cost = NULL;

    if (strpbrk(VersionM,"abcdefghijklmnopqurtsuvwxyz") != NULL) fprintf(stderr, "\n**** THIS IS EXPERIMENTAL CODE ****\n");
    if (strcmp(VersionM,VersionC) != 0) {
        fprintf(stderr, "\n**** MODULE VERSION MISMATCH ****\n");
        fprintf(stderr, "      Main module : %-6s\n",VersionM);
        fprintf(stderr, "      Comm module : %-6s\n",VersionC);
        return 0;
    }

    while (! b && i < argc) if (strcmp(argv[i++],"-b") == 0) b = TRUE;

    if (b) {
        fprintf(stderr, "\n");
        i = 0;
        while (i < argc) fprintf(stderr, "%s ",argv[i++]);
        fprintf(stderr, "\n");
    }

     /* options descriptor */
    static struct option longopts[] = {
        { "address",		required_argument,	0,	'a' },
	{ "last-alarms",	no_argument,		0,	'A' },
        { "verbose",		no_argument,		0,	'b' },
	{ "calc-value",		required_argument,	0,	'C' },
        { "columnize",		no_argument,		0,	'c' },
        { "get-dsp",		optional_argument,	0,	'd' },
        { "get-dsp-extended",	no_argument,		0,	'D' },
        { "get-dsp-3phase",     no_argument,		0,	'E' },
        { "get-energy",		no_argument,		0,	'e' },
        { "firmware-ver",	no_argument,		0,	'f' },
        { "mfg-date",		no_argument,		0,	'g' },
        { "hide-dsp-msg",	no_argument,		0,	'H' },
        { "help",		no_argument,		0,	'h' },
        { "get-count",		no_argument,		0,	'i' },
        { "get-joules",		no_argument,		0,	'j' },
        { "daily-kwh",          optional_argument,      0,      'k' },
        { "adjust-time",        required_argument,	0,	'L' },
	{ "delay",		required_argument,	0,	'l' },
        { "get-conf",		no_argument,		0,      'm' },
        { "serial-number",	no_argument,		0,	'n' },
        { "output-file",        required_argument,      0,      'o' },
        { "part-number",	no_argument,		0,	'p' },
        { "comm-pause",		required_argument,	0,	'P' },
        { "energy-sent",	optional_argument,	0,	'q' },
        { "read-timeout",	required_argument,	0,	'R' },
        { "calc-grid-power",	no_argument,		0,	'r' },
        { "set-time",		no_argument,		0,	'S' },
        { "get-state",		no_argument,		0,	's' },
        { "get-invtime",	no_argument,		0,      't' },
        { "get-loctime",	no_argument,		0,	'T' },
        { "version",		no_argument,		0,	'V' },
        { "inv-version",	no_argument,		0,	'v' },
	{ "read-pause",		required_argument,	0,	'U' },
	{ "rpt-read-pause",	no_argument,		0,	'u' },
        { "lock-wait",		required_argument,	0,	'w' },
        { "rts-cts",		no_argument,		0,	'X' },
	{ "xon-xoff",		no_argument,		0,	'x' },
	{ "retries",		required_argument,	0,	'Y' },
	{ "rpt-retries",	no_argument,		0,	'y' },
        { NULL,				0,				NULL,   0 }
    };

    /* Set command line defaults */
    yDelay = 1;

    if (argc == 1)
        return 0;				/* no parms at all */

    while ((c = getopt_long(argc, argv, "a:AbC:cDd:EefgHhijk:L:l:mno:P:pq:R:rSsTtVvU:uw:XxY:y", longopts, NULL )) != EOF) {
        switch (c) {
            case 'a':
                /* Inverter address */
                yAddress = atoi(optarg);
                break;
            case 'A': bGetLastAlarms     = TRUE; break;
            case 'b': bVerbose     = TRUE; break;
            case 'C':
                SubStrPos = strstr(optarg, ":");
                if (SubStrPos == NULL)
                    yCost = atof(optarg);
                else {
                    Cost = getMemPtr(SubStrPos-optarg);
                    strncpy(Cost,optarg,SubStrPos-optarg);
                    yCost = atof(Cost);
                    sCostType = getMemPtr(strlen(optarg)-(SubStrPos-optarg));
                    strcpy(sCostType,SubStrPos+1);
                    free(Cost);
                    Cost = NULL;
                }
                if (bVerbose) fprintf(stderr, "\nCost: %f Type: \"%s\"\n",yCost,sCostType);
                break;
            case 'c': bColumns     = TRUE; break;
            case 'd':
                if (optarg == NULL) {
                    yGetDSP = 0;
                } else {
                    i = atoi(optarg);
                    if (i < 0 || i > 2) {
                        fprintf(stderr, "\n%s: %s: -d value out of range, 0 to 2.\n",RunTime,ProgramName);
                        return 0;
                    }
                    yGetDSP = i;
                }
                break;
            case 'D': bGetDSPExtended    = TRUE; break;
            case 'E': bGetDSP3Phase      = TRUE; break;
            case 'e': bGetEnergy   = TRUE; break;
            case 'f': bGetVerFW    = TRUE; break;
            case 'g': bGetMfg      = TRUE; break;
            case 'H': bHideDSP     = TRUE; break;
            case 'h': bHelp        = TRUE; break;
            case 'i': bGetCount    = TRUE; break;
            case 'j': bGetJoules   = TRUE; break;
            case 'k':
                /* Get ECC */
                if (optarg == NULL) {
                    yGetEnergyDaily = 366;
                } else {
                    i = atoi(optarg);
                    if (i < 1 || i > 366) {
                        fprintf(stderr, "\n%s: %s: -k value out of range, 1 to 366.\n",RunTime,ProgramName);
                        return 0;
                    }
                    yGetEnergyDaily = i;
                }
                break;
            case 'L':
                i = atoi(optarg);
                if (i < 0) {
                    fprintf(stderr, "\n%s: %s: -L value out of range, 0 or >=1 0 = check for DST change >=1 = Reconcile with computer time\n",RunTime,ProgramName);
                    return 0;
                }
                yCheckSetTime = i;
                break;
            case 'l':
                /* Get delay time */
                i = atoi(optarg);
                if (i < 0 || i > 255) {
                        fprintf(stderr, "\n%s: %s: -l Illegal delay specified.\n",RunTime,ProgramName);
                        return 0;
                }
                yDelay = (unsigned char)i;
                break;
            case 'm': bGetConf     = TRUE; break;
            case 'n': bGetSN       = TRUE; break;
            case 'o':
                outfile = getMemPtr(strlen(optarg)); 
                strcpy(outfile, optarg);
                break;
            case 'p': bGetPN       = TRUE; break;
            case 'P':
                yCommPause = atoi(optarg);
                if (yCommPause <= 0 || yCommPause > 1000000) {
                    fprintf(stderr, "\n%s: %s: -P Comm Pause micro-seconds (%li) out of range, 1-1000000.\n",RunTime,ProgramName,yCommPause);
                    return 0;
                }
                break;
            case 'q':
                /* Get Energy Sent */
                if (optarg == NULL) {
                    yGetEnergySent = 4320;
                } else {
                    i = atoi(optarg);
                    if (i < 1 || i > 1440) {
                        fprintf(stderr, "\n%s: %s: -k value out of range, 1 to 1440.\n",RunTime,ProgramName);
                        return 0;
                    }
                    yGetEnergySent = i*3;
                }
                break;
            case 'R':
                /* read timeout value in us */
                yTimeout = atoi(optarg)*1000;
                break;
            case 'r': bCalcGridPwr = TRUE; break;
            case 'S': bSetTime     = TRUE; break;
            case 's': bGetState    = TRUE; break;
            case 'T': bGetLocTime  = TRUE; break;
            case 't': bGetInvTime  = TRUE; break;
            case 'U':
                yReadPause = atoi(optarg);
                if (yReadPause < 1 || yReadPause > 10000) {
                    fprintf(stderr, "\n%s: %s: -U Read Pause milli-seconds (%d) out of range, 1-10000.\n",RunTime,ProgramName,yReadPause);
                    return 0;
                }
                break;
            case 'u': bRptReadPause	= TRUE; break;
            case 'w':
                yLockWait = atoi(optarg);
                if (yLockWait < 1 || yLockWait > 30) {
                    fprintf(stderr, "\n%s: %s: -w Lock Wait seconds (%d) out of range, 1-30.\n",RunTime,ProgramName,yLockWait);
                    return 0;
                }
                break;
            case 'X': bRTSCTS      = TRUE; break;
            case 'x': bXonXoff     = TRUE; break;
            case 'Y':
                yMaxAttempts = atoi(optarg);
                if (yMaxAttempts < 1 || yMaxAttempts > 100) {
                    fprintf(stderr, "\n%s: %s: -Y Retries (%d) out of range, 1-100.\n",RunTime,ProgramName,yMaxAttempts);
                    return 0;
                }
                break;
            case 'y': bRptReties   = TRUE; break;
            case 'V': bVersion     = TRUE; break;
            case 'v': bGetVer      = TRUE; break;

            case '?': /* user entered unknown option */
            case ':': /* user entered option without required value */
                return 0;

            default:
                break;
        }
    }
    if (optind < argc) { 			/* get serial device name */
        szttyDevice = getMemPtr(strlen(argv[optind]));
        strcpy(szttyDevice, argv[optind]);
     } else {
        if (!bVersion && ! bHelp) fprintf(stderr, "\n%s: %s: No serial device specified\n",RunTime,ProgramName);
        return 0;
    }
    pos = strrchr(szttyDevice, '/');
    if (pos > 0) {
        pos++;
        devLCKfile = getMemPtr(strlen(ttyLCKloc)+(strlen(szttyDevice)-(pos-szttyDevice))+1);
        devLCKfile[0] = '\0';
        strcpy(devLCKfile,ttyLCKloc);
        strcat(devLCKfile, pos);
        sprintf(sPID,"%lu",PID);
        devLCKfileNew = getMemPtr(strlen(devLCKfile)+strlen(sPID)+1);
        devLCKfileNew[0] = '\0';
        strcpy(devLCKfileNew,devLCKfile);
        sprintf(devLCKfileNew,"%s.%lu",devLCKfile,PID);
    } else {
        devLCKfile = NULL;
    }
    if (bVerbose) fprintf(stderr, "\nszttyDevice: %s\nyDelay:  %i\nyTimeout %i us\ndevLCKfile: <%s>\ndevLCKfileNew: <%s>\n",szttyDevice,yDelay,yTimeout,devLCKfile,devLCKfileNew);

    if (bSetTime && yCheckSetTime >= 0) {
        fprintf(stderr, "\n%s: %s: -L and -S are mutually exclusive.\n",RunTime,ProgramName);
        return 0;
    }
    if (yAddress < 1 || yAddress > 63) {
        fprintf(stderr, "\n%s: %s: Illegal address (%d) specified.\n",RunTime,ProgramName,yAddress);
        return 0;
    }
    if (bVerbose) fprintf(stderr, "Got Params\n");
    return -1;
}


/*--------------------------------------------------------------------------
        getMemPtr
----------------------------------------------------------------------------*/
void *getMemPtr(size_t mSize)
{
    void *ptr;

   ptr = malloc(mSize);
   if (!ptr) {
       fprintf(stderr, "\nvproweather: malloc failed\n");
       exit(2);
   }
   return ptr;
}


/*--------------------------------------------------------------------------
    Version
    Display program component versions
----------------------------------------------------------------------------*/
void Version()
{
    printf("\nAurora module versions:\n");
    printf("Main module : %-6s\n",VersionM);
    printf("Comm module : %-6s\n",VersionC);
    printf("main.h      : %-6s\n",VersionMHc);
    printf("comm.h      : %-6s\n",VersionCHc);
    printf("names.h     : %-6s\n",VersionNHc);
    printf("states.h    : %-6s\n\n",VersionSHc);
}
 
