/* Gnome Scan - Scan as easy as you print
 * Copyright © 2007  Étienne Bersac <bersace03@laposte.net>
 *
 * Gnome Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

/**
 * SECTION: gnome-scan-acquisition-dialog
 * @short_description: Give the user the acquisition control
 * @include: gnome-scan.h
 *
 * The acquisition process might be long, the
 * #GnomeScanAcquisitionDialog primarily shows a progress bar and of
 * course, allow user to cancel the job.
 *
 * The #GnomeScanAcquisitionDialog also has nice UI to handle mass
 * acquisition. This is done through a simple UI in order to make it
 * very quick. The dialog shows a stop button which either cancel a
 * running acquisition or end the mass acquisition process. Ideally,
 * the scanner has automatic document feeder and the user just puts
 * its photos/document page and trigger acquisition. However, since
 * most scanner does not have such feature (e.g. flatbed scanner or
 * sheetfed scanner), #GnomeScanAcquisitionDialog shows a Next button
 * that redo the scan with the same configuration. The idea is to
 * allow the user to put page, configure, trigger, put next page,
 * press Enter, and loop until there is not other page.
 *
 * The acquisition and processing is done in a work thread. See
 * #GnomeScanJob and gnome_scan_job_run().
 **/

#include <glib/gi18n.h>
#include "gnome-scan-private.h"
#include "gnome-scan-acquisition-dialog.h"

#define	GET_PRIVATE(o)	(G_TYPE_INSTANCE_GET_PRIVATE ((o), GNOME_TYPE_SCAN_ACQUISITION_DIALOG, GnomeScanAcquisitionDialogPrivate))

typedef struct _GnomeScanAcquisitionDialogPrivate GnomeScanAcquisitionDialogPrivate;

struct _GnomeScanAcquisitionDialogPrivate
{
	GnomeScanJob	*job;
	
	GtkWidget	*progressbar;
	GtkWidget	*state_label;
	GtkWidget	*forward;
	GThread*	job_thread;
	gboolean	disposed;
};

enum
{
	PROP_0,
	PROP_JOB
};


static GObjectClass* parent_class = NULL;

static void	gsad_run (GnomeScanAcquisitionDialog *dialog);
static gboolean	gsad_monitor (GnomeScanAcquisitionDialog *dialog);
static void	gsad_close	(GnomeScanAcquisitionDialog *dialog);
static void	gsad_forward	(GnomeScanAcquisitionDialog *dialog);

G_DEFINE_TYPE (GnomeScanAcquisitionDialog, gnome_scan_acquisition_dialog, GTK_TYPE_DIALOG);

static void
gnome_scan_acquisition_dialog_init (GnomeScanAcquisitionDialog *object)
{
	GnomeScanAcquisitionDialogPrivate *priv = GET_PRIVATE (object);
	GtkWidget *label, *button, *hbox, *icon, *text_box;
	GtkWidget *vbox = GTK_DIALOG (object)->vbox;
	GtkWidget *action_area = GTK_DIALOG (object)->action_area;
	
	/* parent properties */
	g_object_set (object,
				  "title", _("Acquisition"),
				  "window-position", GTK_WIN_POS_CENTER,
				  "has-separator", FALSE,
				  "border-width", 6,
				  "resizable", FALSE,
				  NULL);
	
	g_signal_connect_swapped (object, "delete-event",
							  (GCallback) gsad_close,
							  object);
	
	/* container border */
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 6);
	gtk_box_set_spacing (GTK_BOX (vbox), 6);
	gtk_container_set_border_width (GTK_CONTAINER (GTK_DIALOG (object)->action_area), 6);
	
	
	hbox = gtk_hbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 6);
	
	icon = gtk_image_new_from_icon_name ("scanner", GTK_ICON_SIZE_DIALOG);
	gtk_misc_set_padding (GTK_MISC (icon), 6, 6);
	gtk_box_pack_start (GTK_BOX (hbox), icon, FALSE, TRUE, 0);
	
	text_box = gtk_vbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (hbox), text_box, TRUE, TRUE, 0);
	
	/* primary text */
	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label),
						  g_strconcat ("<span size=\"x-large\"><b>", _("Acquisition"), "</b></span>",NULL));
	gtk_misc_set_alignment (GTK_MISC (label), 0., .5);
	gtk_box_pack_start (GTK_BOX (text_box), label, FALSE, TRUE, 0);
	
	/* secondary text */
	label = gtk_label_new (_("The software now acquires and processes images "
							 "according to the settings."));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_box_pack_start (GTK_BOX (text_box), label, FALSE, TRUE, 0);
	
	/* progress bar */
	priv->progressbar = gtk_progress_bar_new ();
	/*gtk_progress_bar_set_text (GTK_PROGRESS_BAR (priv->progressbar),
	 _("Waiting for device"));*/
	gtk_box_pack_start (GTK_BOX (vbox), priv->progressbar,
						FALSE, TRUE, 0);
	
	/* state label */
	priv->state_label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (priv->state_label),
						  g_strconcat ("<i>", _("Waiting for device"), "</i>", NULL));
	gtk_misc_set_alignment (GTK_MISC (priv->state_label), 0., .5);
	gtk_box_pack_start (GTK_BOX (vbox), priv->state_label, FALSE, TRUE, 0);
	
	/* buttons */
	button = gtk_button_new_from_stock (GTK_STOCK_STOP);
	g_signal_connect_swapped (button, "clicked",
							  (GCallback) gsad_close,
							  object);
	gtk_box_pack_start (GTK_BOX (action_area), button, FALSE, FALSE, 0);
	
	button = priv->forward = gtk_button_new_from_stock (GTK_STOCK_GO_FORWARD);
	g_signal_connect_swapped (button, "clicked",
							  G_CALLBACK (gsad_forward),
							  object);
	gtk_box_pack_start (GTK_BOX (action_area), button, FALSE, FALSE, 0);
}

static void
gnome_scan_acquisition_dialog_dispose (GObject *object)
{
	GnomeScanAcquisitionDialogPrivate *priv = GET_PRIVATE(object);
	
	if (!priv->disposed) {
		g_object_unref(priv->job);
		priv->disposed = TRUE;
	}
	
	G_OBJECT_CLASS (parent_class)->finalize (object);
}
static void
gnome_scan_acquisition_dialog_finalize (GObject *object)
{
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_scan_acquisition_dialog_set_property (GObject *object,
											guint prop_id,
											const GValue *value,
											GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_ACQUISITION_DIALOG (object));
	
	switch (prop_id)
	{
		case PROP_JOB:
			GET_PRIVATE (object)->job = GNOME_SCAN_JOB (g_value_dup_object (value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}

static void
gnome_scan_acquisition_dialog_get_property (GObject *object,
											guint prop_id,
											GValue *value,
											GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_ACQUISITION_DIALOG (object));
	
	switch (prop_id)
	{
		case PROP_JOB:
			g_value_set_object (value, GET_PRIVATE (object)->job);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}

/**
 * gnome_scan_acquisition_dialog_class_init:
 * @klass: 
 * 
 * 
 **/
static void
gnome_scan_acquisition_dialog_class_init (GnomeScanAcquisitionDialogClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS (klass);
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	
	g_type_class_add_private (klass, sizeof (GnomeScanAcquisitionDialogPrivate));
	object_class->finalize = gnome_scan_acquisition_dialog_finalize;
	object_class->finalize = gnome_scan_acquisition_dialog_dispose;
	object_class->set_property = gnome_scan_acquisition_dialog_set_property;
	object_class->get_property = gnome_scan_acquisition_dialog_get_property;
	
	/**
	 * GnomeScanAcquisitionDialog:job:
	 *
	 * The job being monitored by the #GnomeScanAcquisitionDialog.
	 **/
	g_object_class_install_property (object_class,
									 PROP_JOB,
									 g_param_spec_object ("job",
														  "Job",
														  "The job the dialog is handling",
														  GNOME_TYPE_SCAN_JOB,
														  G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


/**
 * gnome_scan_acquisition_dialog_new:
 * @parent: the parent #GtkWindow
 * @job: the #GnomeScanJob to run
 * 
 * Create a new dialog ready to run @job
 * 
 * Returns: a new #GnomeScanAcquisitionDialog
 **/
GtkWidget*
gnome_scan_acquisition_dialog_new (GtkWindow *parent, GnomeScanJob *job)
{
	GObject* object= g_object_new (GNOME_TYPE_SCAN_ACQUISITION_DIALOG,
								   "transient-for", parent,
								   "job", job,
								   NULL);
	return GTK_WIDGET (object);
}

/**
 * gnome_scan_acquisition_dialog_run:
 * @dialog: a #GnomeScanAcquisitionDialog
 * 
 * Start a #GMainLoop, run #GnomeScanAcquisitionDialog:job in a worker
 * thread, monitor progress and allow user to control the execution of
 * the job. Note that the progressbar is updated about 24 times per
 * seconds.
 **/
void
gnome_scan_acquisition_dialog_run (GnomeScanAcquisitionDialog *dialog)
{
	gtk_init_add ((GtkFunction) gsad_run, dialog);
	gtk_main ();
}



/* INTERNALS */

/* ThreadFunc */
static void
gsad_job_run (GnomeScanJob *job)
{
	gnome_scan_job_run (job);
	g_thread_exit (NULL);
}

static void
gsad_run (GnomeScanAcquisitionDialog *dialog)
{
	GnomeScanAcquisitionDialogPrivate *priv = GET_PRIVATE (dialog);
	GError *error = NULL;
	
	gtk_widget_show_all (GTK_WIDGET (dialog));
	gtk_widget_set_sensitive (priv->forward, FALSE);
	/* ~24 updates per second -> update each 42 microseconds */
	g_timeout_add (42, (GSourceFunc) gsad_monitor, dialog);
	gnome_scan_job_configure (priv->job);
	priv->job_thread = g_thread_create ((GThreadFunc) gsad_job_run,
										priv->job, TRUE, &error);
}

static gboolean
gsad_monitor (GnomeScanAcquisitionDialog *dialog)
{
	GnomeScanAcquisitionDialogPrivate *priv = GET_PRIVATE (dialog);
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (priv->progressbar),
								   priv->job->progress);
	gtk_label_set_markup (GTK_LABEL (priv->state_label),
						  g_strdup_printf ("<i>%s</i>", priv->job->stage));
	
	gtk_widget_set_sensitive (priv->forward,
							  priv->job->done);
	if (GTK_WIDGET_CAN_FOCUS(priv->forward))
		gtk_widget_grab_focus (priv->forward);
	return !priv->job->done;
}

/* callbacks */
static void
gsad_close (GnomeScanAcquisitionDialog *dialog)
{
	GnomeScanAcquisitionDialogPrivate *priv = GET_PRIVATE (dialog);
	gtk_widget_hide (GTK_WIDGET (dialog));
	gnome_scan_job_cancel (priv->job);
	g_thread_join (priv->job_thread);
	gtk_main_quit ();
}

static void
gsad_forward (GnomeScanAcquisitionDialog *dialog)
{
	gsad_run (dialog);
}
