/*
 *  This file is part of GNUDoQ, Copyright (C) 2005-2006 Luc Vo Van
 *
 *   GNUDoQ is free software; you can redistribute it and/or modify it
 *   under the terms of the GNU General Public License as published by the
 *   Free Software Foundation; either version 2, or (at your option) any
 *   later version.
 *
 *   GNUDoQ is distributed in the hope that it will be useful, but WITHOUT
 *   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *   for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with GNUDoQ; see the file COPYING.  If not, write to
 *   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *   Boston, MA 02111-1307, USA.
 *
 *   Class          :   Number widget
 *   Author         :   Luc Vo Van
 *   Original Date  :   September 2005
 *
 **/

#ifndef GNUDOQBOXWIDGET_H
#define GNUDOQBOXWIDGET_H

#include <QWidget>
#include <QColor>
#include <QPainterPath>

/**
 * Number input widget. 81 instances of these widgets are used to form the
 * grid. Clicking them changes their values, they can be editable by the
 * player or not (depending on their editable() propery), can be marked
 * erroneous, etc.
 *
 * This widget is also called a 'box'.
 */
class GNUDoQBoxWidget : public QWidget
{
    Q_OBJECT

public:
    /**
     * Class constructor. Each GNUDoQBoxWidget must belong to a group,
     * which is one of the 9 3x3 zones of the grid. The numbers are
     * as follows:
     *
     * 012
     * 345
     * 678
     *
     * @param iGroup Group number to which this box belongs
     * @param parent Parent widget
     */
    GNUDoQBoxWidget(int iGroup,QWidget *parent = 0);
    /**
     * Returns the value of this box.
     *
     * @see setValue(const int)
     * @return the value of this box, 0 if no value is displayed
     */
    int value() { return m_iValue; }

    /*
     * Returns a value indicating wether or not this box is marked erroneous.
     *
     * @see setErroneous(const bool)
     * @return true if the box is erroneous, false otherwise
     */
    bool erroneous() { return m_bErroneous; }

    /**
     * Returns a value indicating wether or not this box is editable by the
     * player.
     *
     * @see setEditable(const bool)
     * @return true if the box is editable, false otherwise
     */
    bool editable() { return m_bEditable; }

    /**
     * Returns a value indicating wether or not this box's value was
     * determined by the GNUDoku solver (after the player presses the
     * 'Solve' button)
     *
     * @see setCpusolved(const bool)
     * @return true if the box's value was determined by the GNUDoku
     *         solver, false otherwise
     */
    bool cpusolved() { return m_bCpusolved; }

public slots:
    /**
     * Sets the value of this box.
     *
     * @param iValue value to give this box (0 if no value is to be displayed)
     * @see value()
     */
    void setValue(const int iValue);

    /**
     * Sets a value indicating if the box contains an erroneous value.
     * 
     * @param bErroneous true to set the box to erroneous, false otherwise
     * @see erroneous()
     */
    void setErroneous(const bool bErroneous);

    /**
     * Sets a value indicating wether or not the box's value can be edited
     * by the player.
     * 
     * @param bEditable true to make the box player editable, false otherwise
     * @see editable()
     */
    void setEditable(const bool bEditable);

    /**
     * Sets a value indicating wether or not the box's value was determined
     * by the GNUDoku solver.
     *
     * @param bCpuSolved true if the value was determined by the GNUDoku solver
     *        false otherwise (value set by the player)
     * @see cpusolved()
     */
    void setCpusolved(const bool bCpusolved);

signals:
    /**
     * Emitted whenever the value of this box changes
     *
     * @see setValue(const int)
     */
    void valueChanged();

protected:
    /**
     * Draws the box
     * @param event QPaintEvent
     */
    void paintEvent(QPaintEvent *event);

    /**
     * Reacts to keypresses: numbers set values, backspace/del removes
     * the current value (sets it to 0).
     *
     * @param event QKeyEvent
     * @see setValue(const int)
     */
    void keyPressEvent(QKeyEvent *event);

    /**
     * Reacts to clicks, left click increases the value, right click
     * decreases, middle click remove the current value (sets it to 0)
     *
     * @param event QMouseEvent
     * @see setValue(const int)
     */
    void mousePressEvent(QMouseEvent *event);

    /**
     * Reacts to mouse wheel events. Rolling up increases the value, rolling
     * down decreases it.
     *
     * @param event QWheelEvent
     * @see setValue(const int)
     */
    void wheelEvent(QWheelEvent * event);

    /**
     * Sets m_bHovered to true, to highlight the box when hovered by the
     * mouse cursor.
     *
     * @param event QEvent
     */
    void enterEvent(QEvent *event);

    /**
     * Sets m_bHovered to false, to dehighlight the box when not hovered by
     * the mouse cursor anymore.
     *
     * @param event QEvent
     */    
    void leaveEvent(QEvent *event);

private:
    // Properties value holders
    int m_iValue;
    int m_iGroup;
    bool m_bHovered;
    bool m_bErroneous;
    bool m_bEditable;
    bool m_bCpusolved;

    // Colors for each group (3x3 zones)
    static const QColor groupColor[];
};

#endif
