/*
 *  @(#)JUnitTestListener.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.autodoc.v1.testserver.junit;


import junit.framework.Test;
import junit.framework.TestListener;
import junit.framework.AssertionFailedError;

import org.apache.log4j.Logger;

import net.sourceforge.groboutils.autodoc.v1.testserver.MonitorFinder;
import net.sourceforge.groboutils.autodoc.v1.testserver.TestCorrelate;
import net.sourceforge.groboutils.autodoc.v1.testserver.TestData;
import net.sourceforge.groboutils.autodoc.v1.testserver.TestInfo;


/**
 * A helper class to interface between the test framework and JUnit's
 * <tt>TestListener</tt> interface.
 *
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since      March 17, 2002
 * @version    $Date: 2003/06/10 01:08:41 $
 */
public abstract class JUnitTestListener extends TestCorrelate
        implements TestListener
{
    private static final Logger LOG = Logger.getLogger( JUnitTestListener.class );
    
    private MonitorFinder finder;
    
    
    /**
     * 
     */
    public JUnitTestListener( MonitorFinder finder )
    {
        super( null, finder );
    }
    
    
    //-------------------------------------------------------------------------
    // abstract methods
    
    
    /**
     * Called before the test starts, and after the test has been registered.
     */
    protected abstract void startTest( TestData data );
    
    
    /**
     * Called after the test ends.  The implemented method must not send off
     * the data - that is done by the JUnitTestListener implementation.
     */
    protected abstract void endTest( TestData data );
    
    
    /**
     * Called when the test in the data has an unexpected exception thrown.
     */
    protected abstract void addError( TestData data, Throwable t );
    
    
    /**
     * Called when the test in the data has an assertion error.
     */
    protected abstract void addFailure( TestData data, AssertionFailedError t );
    
    
    
    //-------------------------------------------------------------------------
    // JUnit listener methods
    
    /**
     * An error occurred.
     */
    public void addError(Test test, Throwable t)
    {
        if (test == null)
        {
            LOG.error("JUnit passed null test to method addError()",t);
            return;
        }
        
        TestData td = getTestData( createTestInfo( test ) );
        // null means the test was never registered.
        if (td != null)
        {
            addError( td, t );
        }
        else
        {
            LOG.warn("JUnit called 'addError' without calling 'startTest'.",
                t );
        }
    }
    
    /**
     * A failure occurred.
     */
    public void addFailure(Test test, AssertionFailedError t)
    {
        if (test == null)
        {
            LOG.error("JUnit passed null test to method addFailure()",t);
            return;
        }
        
        TestData td = getTestData( createTestInfo( test ) );
        // null means the test was never registered.
        if (td != null)
        {
            addFailure( td, t );
        }
        else
        {
            LOG.warn("JUnit called 'addFailure' without calling 'startTest'.",
                t );
        }
    }
    
    
    /**
     * A test ended.
     */
    public void endTest(Test test)
    {
        TestInfo ti = createTestInfo( test );
        TestData td = getTestData( ti );
        if (td != null)
        {
            endTest( td );
            
            // end off the test info
            try
            {
                getFinder().getMonitor().sendTestData( ti );
            }
            catch (IllegalStateException ise)
            {
                LOG.warn( "Warning for test ["+test+"]: "+ise.getMessage(),
                    ise );
            }
        }
        else
        {
            LOG.warn( "Received an end message for test ["+test+
                "], but it was never added." );
        }
    }
    
    
    /**
     * A test started.
     */
    public void startTest(Test test)
    {
        TestInfo ti = createTestInfo( test );
        
        // start the test info
        getFinder().getMonitor().addTestData( ti );
        
        startTest( getTestData( ti ) );
    }
    
    
    /**
     * Creates a JUnitTestInfo instance.
     */
    protected TestInfo createTestInfo( Test test )
    {
        TestInfo ti = new JUnitTestInfo( test );
        return ti;
    }
}

