/************************************************************************/
/*                                                                      */
/*    vspline - a set of generic tools for creation and evaluation      */
/*              of uniform b-splines                                    */
/*                                                                      */
/*            Copyright 2015 - 2018 by Kay F. Jahnke                    */
/*                                                                      */
/*    The git repository for this software is at                        */
/*                                                                      */
/*    https://bitbucket.org/kfj/vspline                                 */
/*                                                                      */
/*    Please direct questions, bug reports, and contributions to        */
/*                                                                      */
/*    kfjahnke+vspline@gmail.com                                        */
/*                                                                      */
/*    Permission is hereby granted, free of charge, to any person       */
/*    obtaining a copy of this software and associated documentation    */
/*    files (the "Software"), to deal in the Software without           */
/*    restriction, including without limitation the rights to use,      */
/*    copy, modify, merge, publish, distribute, sublicense, and/or      */
/*    sell copies of the Software, and to permit persons to whom the    */
/*    Software is furnished to do so, subject to the following          */
/*    conditions:                                                       */
/*                                                                      */
/*    The above copyright notice and this permission notice shall be    */
/*    included in all copies or substantial portions of the             */
/*    Software.                                                         */
/*                                                                      */
/*    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND    */
/*    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   */
/*    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          */
/*    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       */
/*    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      */
/*    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      */
/*    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     */
/*    OTHER DEALINGS IN THE SOFTWARE.                                   */
/*                                                                      */
/************************************************************************/

/*! \file common.h

    \brief definitions common to all files in this project, utility code
    
    This file contains
    
    - some common enums and strings
    
    - definition of a few utility types used throughout vspline
    
    - exceptions used throughout vspline
    
    It includes vspline/vector.h which defines vspline's use of
    vectorization (meaning SIMD operation) and associated types ans code. 
*/

#ifndef VSPLINE_COMMON
#define VSPLINE_COMMON

#include <vigra/multi_array.hxx>
#include <vigra/tinyvector.hxx>

namespace vspline
{
/// This enumeration is used for codes connected to boundary conditions.
/// There are two aspects to boundary conditions: During prefiltering,
/// the initial causal and anticausal coefficients have to be calculated
/// in a way specific to the chosen boundary conditions. Bracing also needs
/// these codes to pick the appropriate extrapolation code to extrapolate
/// the coefficients beyond the core array.

typedef enum { 
  MIRROR ,    // mirror on the bounds, so that f(-x) == f(x)
  PERIODIC,   // periodic boundary conditions
  REFLECT ,   // reflect, so  that f(-1) == f(0) (mirror between bounds)
  NATURAL,    // natural boundary conditions, f(-x) + f(x) == 2 * f(0)
  CONSTANT ,  // clamp. used for framing, with explicit prefilter scheme
  ZEROPAD ,   // used for boundary condition, bracing
  GUESS ,     // used instead of ZEROPAD to keep margin errors lower
  INVALID
} bc_code;

/// bc_name is for diagnostic output of bc codes

const std::string bc_name[] =
{
  "MIRROR   " ,
  "PERIODIC ",
  "REFLECT  " ,
  "NATURAL  ",
  "CONSTANT " ,
  "ZEROPAD  " ,
  "GUESS    "
} ;

// if 'USE_QUAD' is defined, the literals in poles.h will be interpreted
// as quad literals (with 'Q' suffix), and xlf_type will be __float128.
// otherwise, the suffix used will be L and xlf_type is long double.
// why 'XLF'? it's for eXtra Large Float'.

// TODO: on my system, I tried to compile with
// g++ -std=gnu++11 -fext-numeric-literals -DUSE_QUAD but I got
// 'error: unable to find numeric literal operator ‘operator""Q’'
// I leave this issue open, hoping that using quad literals will
// become unproblematic with future g++ releases. The machinery to
// use them is all in place and ready, and the literals in poles.h
// are precise enough to be used for even higher precision, with
// 64 postcomma digits. should be reasonably future-proof ;)

#ifdef USE_QUAD

#define XLF(arg) (arg##Q)
typedef __float128 xlf_type ;

#else

#define XLF(arg) (arg##L)
typedef long double xlf_type ;

#endif // #ifdef USE_QUAD
  
/// using definition for the 'elementary type' of a type via vigra's
/// ExpandElementResult mechanism. Since this is a frequently used idiom
/// in vspline but quite a mouthful, here's an abbreviation:

template < typename T >
using ET =
typename vigra::ExpandElementResult < T > :: type ;

/// produce a std::integral_constant from the size obtained from
/// vigra's ExpandElementResult mechanism

template < typename T >
using EN =
typename
std::integral_constant
  < int ,
    vigra::ExpandElementResult < T > :: size
  > :: type ;

/// is_element_expandable tests if a type T is known to vigra's
/// ExpandElementResult mechanism. If this is so, the type is
/// considered 'element-expandable'.
  
template < class T >
using is_element_expandable = typename
  std::integral_constant
  < bool ,
    ! std::is_same
      < typename vigra::ExpandElementResult < T > :: type ,
        vigra::UnsuitableTypeForExpandElements
      > :: value
  > :: type ;

/// produce the 'canonical type' for a given type T. If T is
/// single-channel, this will be the elementary type itself,
/// otherwise it's a TinyVector of the elementary type.
/// optionally takes the number of elements the resulting
/// type should have, to allow construction from a fundamental
/// and a number of channels.
  
// vspline 'respects the singular'. this is to mean that throughout
// vspline I avoid using fixed-size containers holding a single value
// and use the single value itself. While this requires some type
// artistry in places, it makes using the code more natural. Only
// when it comes to providing generic code to handle data which may
// or may not be aggregates, vspline moves to using 'synthetic' types,
// which are always TinyVectors, possibly with only one element.
// user code can pass data as canonical or synthetic types. All
// higher-level operations will produce the same type as output
// as they receive as input. The move to 'synthetic' types is only
// done internally to ease the writing of generic code.
// both the 'canonical' and the 'synthetic' type have lost all
// 'special' meaning of their components (like their meaning as
// the real and imaginary part of a complex number). vspline will
// process all types in it's scope as such 'neutral' aggregates of
// several identically-typed elementary values.

template < typename T , int N = EN < T > :: value >
using canonical_type =
typename
  std::conditional
  < N == 1 ,
    ET < T > ,
    vigra::TinyVector < ET < T > , N >
  > :: type ;

template < typename T , int N = EN < T > :: value >
using synthetic_type =
vigra::TinyVector < ET < T > , N > ;

template < class T , size_t sz = 1 >
struct invalid_scalar
{
  static_assert ( sz == 1 , "scalar values can only take size 1" ) ;
} ;

/// definition of a scalar with the same template argument list as
/// a simdized type, to use 'scalar' in the same syntactic slot

template < class T , size_t sz = 1 >
using scalar =
  typename std::conditional
    < sz == 1 ,
      T ,
      invalid_scalar < T , sz >
    > :: type ;

/// vspline creates vigra::MultiArrays of vectorized types. As long as
/// the vectorized types are Vc::SimdArray or vspline::simd_tv, using
/// std::allocator is fine, but when using other types, using a specific
/// allocator may be necessary. Currently this is never the case, but I
/// have the lookup of allocator type from this traits class in place if
/// it should become necessary.

template < typename T >
struct allocator_traits
{
  typedef std::allocator < T > type ;
} ;

// TODO My use of exceptions is a bit sketchy...

/// for interfaces which need specific implementations we use:

struct not_implemented
: std::invalid_argument
{
  not_implemented ( const char * msg )
  : std::invalid_argument ( msg ) { }  ;
} ;

/// dimension-mismatch is thrown if two arrays have different dimensions
/// which should have the same dimensions.

struct dimension_mismatch
: std::invalid_argument
{
  dimension_mismatch ( const char * msg )
  : std::invalid_argument ( msg ) { }  ;
} ;

/// shape mismatch is the exception which is thrown if the shapes of
/// an input array and an output array do not match.

struct shape_mismatch
: std::invalid_argument
{
  shape_mismatch  ( const char * msg )
  : std::invalid_argument ( msg ) { }  ;
} ;

/// exception which is thrown if an opertion is requested which vspline
/// does not support

struct not_supported
: std::invalid_argument
{
  not_supported  ( const char * msg )
  : std::invalid_argument ( msg ) { }  ;
} ;

/// out_of_bounds is thrown by mapping mode REJECT for out-of-bounds coordinates
/// this exception is left without a message, it only has a very specific application,
/// and there it may be thrown often, so we don't want anything slowing it down.

struct out_of_bounds
{
} ;

/// exception which is thrown when assigning an rvalue which is larger than
/// what the lvalue can hold

struct numeric_overflow
: std::invalid_argument
{
  numeric_overflow  ( const char * msg )
  : std::invalid_argument ( msg ) { }  ;
} ;

} ; // end of namespace vspline

// with these common definitions done, we now include 'vector.h', which
// defines vectorization methods used throughout vspline.

#include "vector.h"

#endif // VSPLINE_COMMON
